package ru.yandex.solomon.model.timeseries.decim;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class DecimPolicy {
    public static final DecimPolicy EMPTY = new DecimPolicy(new DecimPolicyItem[0]);

    @Nonnull
    private final DecimPolicyItem[] items;

    DecimPolicy(DecimPolicyItem[] items) {
        this.items = items;

        for (int i = 1; i < items.length; ++i) {
            DecimPolicyItem current = items[i];
            DecimPolicyItem prev = items[i - 1];
            if (!DecimPolicyItem.isStrictlyAbove(current, prev)) {
                throw new IllegalArgumentException("must be above: " + current + ", " + prev);
            }
        }
    }

    public static DecimPolicyBuilder newBuilder() {
        return new DecimPolicyBuilder();
    }

    DecimPolicy(long ageMillis, long stepMillis) {
        this(new DecimPolicyItem[] { new DecimPolicyItem(ageMillis, Math.toIntExact(stepMillis)) });
    }

    @Nonnull
    public DecimPolicyItem[] getItems() {
        return items;
    }

    public boolean isEmpty() {
        return items.length == 0;
    }

    public long getDecimBefore(long now) {
        if (items.length == 0) {
            return 0;
        }

        var lastApplyPolicy = items[0];
        return lastApplyPolicy.getDecimBefore(now);
    }

    public long getDecimFrom(long decimatedAt) {
        if (items.length == 0 || decimatedAt == 0) {
            return 0;
        }

        var firstApplyPolicy = items[items.length - 1];
        return firstApplyPolicy.getDecimBefore(decimatedAt);
    }

    @Nullable
    public DecimPolicyItem findOldest(long beginMillis, long nowMillis) {
        for (int i = items.length - 1; i >= 0; i--) {
            var item = items[i];
            var decimBefore = item.getDecimBefore(nowMillis);
            if (beginMillis <= decimBefore) {
                return item;
            }
        }
        return null;
    }

    @Override
    public String toString() {
        return "DecimPolicy{" +
                "items=" + Arrays.toString(items) +
                '}';
    }
}
