package ru.yandex.solomon.model.timeseries.decim;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class DecimPolicyItem {
    public static final DecimPolicyItem ZERO = new DecimPolicyItem();

    private final long ageMillis;
    private final int stepMillis;

    private DecimPolicyItem() {
        this.ageMillis = 0;
        this.stepMillis = 0;
    }

    public DecimPolicyItem(long ageMillis, int stepMillis) {
        if (ageMillis <= 0) {
            throw new IllegalArgumentException("Too small ageMillis: " + DurationUtils.formatDurationMillis(ageMillis));
        }
        if (stepMillis <= 0) {
            throw new IllegalArgumentException("Too small stepMillis: " + DurationUtils.formatDurationMillis(stepMillis));
        }
        if (stepMillis % 1000 != 0) {
            throw new IllegalArgumentException(DurationUtils.formatDurationMillis(stepMillis));
        }
        if (ageMillis % stepMillis != 0) {
            throw new IllegalArgumentException("Not valid step " +
                    DurationUtils.formatDurationMillis(stepMillis)
                    + " for age "
                    + DurationUtils.formatDurationMillis(ageMillis));
        }

        this.ageMillis = ageMillis;
        this.stepMillis = stepMillis;
    }

    public long getAgeMillis() {
        return ageMillis;
    }

    public int getStepMillis() {
        return stepMillis;
    }

    public long getDecimBefore(long now) {
        return roundTs(now - ageMillis, stepMillis);
    }

    private static long roundTs(long instantMillis, long stepMillis) {
        return instantMillis - instantMillis % stepMillis;
    }

    public static boolean isStrictlyAbove(DecimPolicyItem a, DecimPolicyItem b) {
        return a.ageMillis > b.ageMillis && a.stepMillis > b.stepMillis;
    }

    @Override
    public String toString() {
        return DurationUtils.formatDurationMillis(ageMillis)
            + ":" + DurationUtils.formatDurationMillis(stepMillis);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        DecimPolicyItem that = (DecimPolicyItem) o;

        if (ageMillis != that.ageMillis) return false;
        return stepMillis == that.stepMillis;

    }

    @Override
    public int hashCode() {
        int result = Long.hashCode(ageMillis);
        result = 31 * result + stepMillis;
        return result;
    }
}
