package ru.yandex.solomon.model.timeseries.decim;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DecimatingAggrGraphDataIterator extends AggrGraphDataListIterator {
    private final AggrGraphDataListIterator source;
    private final DecimPolicy policy;
    private final long now;
    private final DecimPointValueCollector decimCollector;
    private int activePolicy;
    private long decimBefore;
    private final AggrPoint temp = new AggrPoint();

    private DecimatingAggrGraphDataIterator(DecimPointValueCollector decimCollector, AggrGraphDataListIterator source, DecimPolicy policy, long now) {
        super(source.columnSetMask());
        this.source = source;
        this.policy = policy;
        this.now = now;
        this.decimCollector = decimCollector;
        this.activePolicy = policy.getItems().length - 1;
        this.decimBefore = policy.getItems()[activePolicy].getDecimBefore(now);
    }

    public static AggrGraphDataListIterator of(MetricType type, AggrGraphDataListIterator source, DecimPolicy policy, long now) {
        if (policy.getItems().length == 0) {
            return source;
        }

        if (source.estimatePointsCount() == 0) {
            return source;
        }

        DecimPointValueCollector decimCollector = DecimPointValueCollector.of(type);
        return new DecimatingAggrGraphDataIterator(decimCollector, source, policy, now);
    }

    @Override
    public boolean next(AggrPoint target) {
        while (hasNext()) {
            if (!shouldBeDecimated(temp)) {
                temp.copyTo(target);
                temp.tsMillis = TsColumn.DEFAULT_VALUE;
                return true;
            }

            if (decim(target)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public int estimatePointsCount() {
        return source.estimatePointsCount();
    }

    private boolean hasNext() {
        if (temp.tsMillis != TsColumn.DEFAULT_VALUE) {
            return true;
        }

        return source.next(temp);
    }

    private boolean decim(AggrPoint target) {
        DecimPolicyItem policyItem = policy.getItems()[activePolicy];
        long windowStart = temp.tsMillis - (temp.tsMillis % policyItem.getStepMillis());
        long windowEnd = windowStart + policyItem.getStepMillis();

        decimCollector.reset();
        fillBuffer(windowEnd);
        if (!TsColumn.isValid(windowStart)) {
            return false;
        }

        int count = decimCollector.getCount();
        if (count == 0) {
            return false;
        }

        boolean result = decimCollector.compute(target);
        target.tsMillis = windowStart;
        target.stepMillis = Math.max(target.stepMillis, policyItem.getStepMillis());
        return result;
    }

    private void fillBuffer(long windowEnd) {
        do {
            if (temp.tsMillis >= windowEnd) {
                return;
            }

            decimCollector.append(temp);
        } while (source.next(temp));
        temp.tsMillis = TsColumn.DEFAULT_VALUE;
    }

    private boolean shouldBeDecimated(AggrPoint target) {
        do {
            if (target.tsMillis < decimBefore) {
                return true;
            }

            if (activePolicy == 0) {
                return false;
            }

            activePolicy--;
            DecimPolicyItem policyItem = policy.getItems()[activePolicy];
            this.decimBefore = policyItem.getDecimBefore(now);
        } while (true);
    }
}
