package ru.yandex.solomon.model.timeseries.decim;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SimpleDecimationIterator extends AggrGraphDataListIterator {
    private final long stepMillis;
    private final AggrGraphDataListIterator source;
    private final DecimPointValueCollector collector;
    private final AggrPoint temp = new AggrPoint();

    public SimpleDecimationIterator(DecimPointValueCollector collector, long stepMillis, AggrGraphDataListIterator source) {
        super(source.columnSet);
        this.collector = collector;
        this.stepMillis = stepMillis;
        this.source = source;
    }

    public static AggrGraphDataListIterator of(MetricType type, long stepMillis, AggrGraphDataListIterator source) {
        var collector = DecimPointValueCollector.of(type);
        return new SimpleDecimationIterator(collector, stepMillis, source);
    }

    @Override
    public int estimatePointsCount() {
        return source.estimatePointsCount();
    }

    private boolean hasNext() {
        if (temp.tsMillis != TsColumn.DEFAULT_VALUE) {
            return true;
        }

        return source.next(temp);
    }

    private boolean decim(AggrPoint target) {
        long windowStart = temp.tsMillis - (temp.tsMillis % stepMillis);
        long windowEnd = windowStart + stepMillis;

        collector.reset();
        fillBuffer(windowEnd);
        if (!TsColumn.isValid(windowStart)) {
            return false;
        }

        int count = collector.getCount();
        if (count == 0) {
            return false;
        }

        boolean result = collector.compute(target);
        target.tsMillis = windowStart;
        target.stepMillis = Math.max(target.stepMillis, stepMillis);
        return result;
    }

    private void fillBuffer(long windowEnd) {
        do {
            if (temp.tsMillis >= windowEnd) {
                return;
            }

            collector.append(temp);
        } while (source.next(temp));
        temp.tsMillis = TsColumn.DEFAULT_VALUE;
    }

    @Override
    public boolean next(AggrPoint target) {
        while (hasNext()) {
            if (decim(target)) {
                return true;
            }
        }
        return false;
    }
}
