package ru.yandex.solomon.model.timeseries.iterator;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class GenericCombineIterator<C extends GenericCursor, T> implements GenericIterator<T> {
    private final List<C> cursors;
    private final GenericPointCollector<C, T> collector;
    private long latestTsMillis = 0;

    public GenericCombineIterator(List<C> cursors, GenericPointCollector<C, T> collector) {
        this.cursors = cursors;
        this.collector = collector;
    }

    public List<C> getCursors() {
        return cursors;
    }

    private boolean hasNext(C cursor) {
        if (cursor.getTimestamp() > latestTsMillis) {
            return true;
        }

        return cursor.next();
    }

    @Override
    public boolean next(T target) {
        long nextTsMillis = Long.MAX_VALUE;
        boolean hasNext = false;
        int cursorIndex = -1;
        for (var cursor : cursors) {
            cursorIndex++;

            if (!hasNext(cursor)) {
                continue;
            }

            hasNext = true;
            if (nextTsMillis > cursor.getTimestamp()) {
                nextTsMillis = cursor.getTimestamp();
                collector.reset();
            }

            if (nextTsMillis != cursor.getTimestamp()) {
                continue;
            }

            collector.append(cursorIndex, cursor);
        }

        if (!hasNext) {
            return false;
        }

        collector.compute(nextTsMillis, target);
        collector.reset();
        latestTsMillis = nextTsMillis;
        return true;
    }

    @Override
    public int estimatePointsCount() {
        return getCursors().stream()
                .map(c -> c.getIterator().estimatePointsCount())
                .max(Integer::compareTo)
                .orElse(-1);
    }
}
