package ru.yandex.solomon.model.type;

import java.util.Arrays;

import javax.annotation.Nullable;

import io.netty.util.Recycler;

import ru.yandex.monlib.metrics.histogram.Doubles;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
public class Histogram {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Histogram.class)
            + MemoryCounter.arraySize(Histograms.MAX_BUCKETS_COUNT, MemoryCounter.DOUBLE_SIZE)
            + MemoryCounter.arraySize(Histograms.MAX_BUCKETS_COUNT, MemoryCounter.LONG_SIZE);

    private static final Recycler<Histogram> RECYCLE = new Recycler<>() {
        @Override
        protected Histogram newObject(Handle<Histogram> handle) {
            return new Histogram(handle);
        }
    };

    private final Recycler.Handle<Histogram> recycleHandler;
    private final double[] bounds;
    private final long[] buckets;
    private long denom;
    private int size;

    private Histogram(Recycler.Handle<Histogram> recycleHandler) {
        this.recycleHandler = recycleHandler;
        this.bounds = new double[Histograms.MAX_BUCKETS_COUNT];
        this.buckets = new long[Histograms.MAX_BUCKETS_COUNT];
        this.size = 0;
    }

    public static Histogram newInstance() {
        return RECYCLE.get();
    }

    public static Histogram newInstance(double[] bounds, long[] buckets) {
        return newInstance().copyFrom(bounds, buckets);
    }

    public static Histogram copyOf(Histogram histogram) {
        return newInstance().copyFrom(histogram);
    }

    public static Histogram copyOf(HistogramSnapshot snapshot) {
        return newInstance().copyFrom(snapshot);
    }

    public static Histogram orNew(@Nullable Histogram histogram) {
        if (histogram != null) {
            histogram.reset();
            return histogram;
        }

        return newInstance();
    }

    public Histogram copyFrom(Histogram target) {
        System.arraycopy(target.bounds, 0, bounds, 0, target.size);
        System.arraycopy(target.buckets, 0, buckets, 0, target.size);
        size = target.size;
        denom = target.denom;
        return this;
    }

    public Histogram copyFrom(HistogramSnapshot snapshot) {
        for (int index = 0; index < snapshot.count(); index++) {
            bounds[index] = snapshot.upperBound(index);
            buckets[index] = snapshot.value(index);
        }
        size = snapshot.count();
        denom = ValueColumn.DEFAULT_DENOM;
        return this;
    }

    public Histogram copyFrom(double[] bounds, long[] buckets) {
        if (bounds.length != buckets.length) {
            throw new IllegalArgumentException("boundsSize(" + bounds.length + ") != bucketsSize(" + buckets.length + ") ");
        }

        return copyFrom(bounds, buckets, bounds.length);
    }

    public Histogram copyFrom(double[] bounds, long[] buckets, int size) {
        if (size > Histograms.MAX_BUCKETS_COUNT) {
            throw new IllegalArgumentException("boundsCount must <= " + Histograms.MAX_BUCKETS_COUNT + ", got: " + size);
        }
        System.arraycopy(bounds, 0, this.bounds, 0, size);
        System.arraycopy(buckets, 0, this.buckets, 0, size);
        this.size = size;
        return this;
    }

    public Histogram setUpperBound(int index, double value) {
        if (index >= Histograms.MAX_BUCKETS_COUNT) {
            throw new IndexOutOfBoundsException(index);
        }

        bounds[index] = value;
        size = Math.max(size, index + 1);
        return this;
    }

    public Histogram setBucketValue(int index, long value) {
        if (index >= Histograms.MAX_BUCKETS_COUNT) {
            throw new IndexOutOfBoundsException(index);
        }

        buckets[index] = value;
        size = Math.max(size, index + 1);
        return this;
    }

    public double[] getBounds() {
        return bounds;
    }

    public void reset() {
        Arrays.fill(buckets, 0);
        Arrays.fill(bounds, 0);
        this.size = 0;
    }

    public void recycle() {
        reset();
        recycleHandler.recycle(this);
    }

    /**
     * @return buckets count.
     */
    public int count() {
        return size;
    }

    /**
     * @return upper bound for bucket with particular index.
     */
    public double upperBound(int index) {
        if (index < 0 || index >= size) {
            throw new ArrayIndexOutOfBoundsException(index);
        }

        return bounds[index];
    }

    public void addValue(double value, long count) {
        int idx = ru.yandex.monlib.metrics.histogram.Arrays.lowerBound(bounds, size, value);
        buckets[idx] += count;
    }

    /**
     * @return value stored in bucket with particular index.
     */
    public long value(int index) {
        if (index < 0 || index >= size) {
            throw new ArrayIndexOutOfBoundsException(index);
        }

        return buckets[index];
    }

    public long getDenom() {
        return denom;
    }

    public double valueDivided(int index) {
        long num = value(index);
        return ValueColumn.divide(num, denom);
    }

    public Histogram setDenom(long denom) {
        if (!ValueColumn.isValidDenom(denom)) {
            throw new RuntimeException("Invalid denom: " + denom);
        }
        this.denom = denom;
        return this;
    }

    public boolean boundsEquals(Histogram rhs) {
        if (count() != rhs.count()) {
            return false;
        }

        return Arrays.equals(bounds, 0, size, rhs.bounds, 0, rhs.size);
    }

    public boolean bucketsEquals(Histogram rhs) {
        if (count() != rhs.count()) {
            return false;
        }

        return Arrays.equals(buckets, 0, size, rhs.buckets, 0, rhs.size);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Histogram)) {
            return false;
        }
        Histogram snapshot = (Histogram) o;
        if (this.denom != snapshot.denom) return false;
        return boundsEquals(snapshot) && bucketsEquals(snapshot);
    }

    @Override
    public int hashCode() {
        int result = Arrays.hashCode(buckets);
        result = 31 * result + Arrays.hashCode(bounds);
        result = 31 * result + (int) (denom ^ (denom >>> 32));
        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        int i = 0;
        for (; i < count() - 1; i++) {
            sb.append(Doubles.toString(upperBound(i))).append(": ").append(value(i));
            sb.append(", ");
        }

        if (count() != i) {
            if (Double.compare(upperBound(i), Histograms.INF_BOUND) >= 0) {
                sb.append("inf: ").append(value(i));
            } else {
                sb.append(Doubles.toString(upperBound(i))).append(": ").append(value(i));
            }
        }
        sb.append('}');
        return sb.toString();
    }
}
