package ru.yandex.solomon.model.type;

import java.util.Arrays;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.util.Recycler;

import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class LogHistogram {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LogHistogram.class)
            + MemoryCounter.arraySize(LogHistogram.LIMIT_MAX_BUCKETS_SIZE, MemoryCounter.DOUBLE_SIZE);

    public static final Recycler<LogHistogram> RECYCLE = new Recycler<>() {
        @Override
        protected LogHistogram newObject(Handle<LogHistogram> handle) {
            return new LogHistogram(handle);
        }
    };
    public static final int DEFAULT_MAX_BUCKET_SIZE = 100;
    public static final double DEFAULT_BASE = 1.5;

    /**
     * Absolute maximum for count buckets in histogram, it's necessary limit to prevent waste a memory.
     */
    public static final int LIMIT_MAX_BUCKETS_SIZE = 100;

    private final Recycler.Handle<LogHistogram> recycleHandler;
    private final double[] buckets;
    private long countZero = 0;
    private int startPower = 0;
    private int maxBucketsSize = DEFAULT_MAX_BUCKET_SIZE;
    private double base = DEFAULT_BASE;
    private int size;

    public LogHistogram(Recycler.Handle<LogHistogram> recycleHandler) {
        this.recycleHandler = recycleHandler;
        this.buckets = new double[DEFAULT_MAX_BUCKET_SIZE];
    }

    public static LogHistogram newBuilder() {
        return newInstance();
    }

    public static LogHistogram newInstance() {
        return RECYCLE.get();
    }

    public static LogHistogram copyOf(LogHistogram histogram) {
        LogHistogram result = newInstance();
        result.copyFrom(histogram);
        return result;
    }

    public static LogHistogram orNew(@Nullable LogHistogram histogram) {
        if (histogram == null) {
            return newInstance();
        }

        histogram.reset();
        return histogram;
    }

    public static LogHistogram ofBuckets(double... buckets) {
        return newBuilder()
            .setBuckets(buckets)
            .build();
    }

    public void reset() {
        Arrays.fill(buckets, 0);
        countZero = 0;
        startPower = 0;
        maxBucketsSize = DEFAULT_MAX_BUCKET_SIZE;
        base = DEFAULT_BASE;
        size = 0;
    }

    public LogHistogram copyFrom(LogHistogram histogram) {
        System.arraycopy(histogram.buckets, 0, buckets, 0, histogram.size);
        size = histogram.size;
        base = histogram.base;
        maxBucketsSize = histogram.maxBucketsSize;
        countZero = histogram.countZero;
        startPower = histogram.startPower;
        return this;
    }

    public void recycle() {
        reset();
        recycleHandler.recycle(this);
    }

    public int countBucket() {
        return size;
    }

    public double getBucketValue(int index) {
        if (index < 0 || index >= size) {
            throw new ArrayIndexOutOfBoundsException(index);
        }

        return buckets[index];
    }

    public long getCountZero() {
        return countZero;
    }

    public LogHistogram setCountZero(long countZero) {
        this.countZero = countZero;
        return this;
    }

    public int getStartPower() {
        return startPower;
    }

    public LogHistogram setStartPower(int startPower) {
        this.startPower = startPower;
        return this;
    }

    public int getMaxBucketsSize() {
        return maxBucketsSize;
    }

    public LogHistogram setMaxBucketsSize(int maxBucketsSize) {
        if (maxBucketsSize > LIMIT_MAX_BUCKETS_SIZE) {
            throw new IllegalArgumentException(
                "Max count bucket can't be more than " + LIMIT_MAX_BUCKETS_SIZE + ", but specified " + maxBucketsSize
            );
        }
        this.maxBucketsSize = maxBucketsSize;
        return this;
    }

    public double getBase() {
        return base;
    }

    public LogHistogram setBase(double base) {
        if (Double.isNaN(base) || Double.isInfinite(base)) {
            throw new IllegalArgumentException("Base can't be infinite or nan, specified: " + base);
        }
        this.base = base;
        return this;
    }

    public double[] copyBuckets() {
        return Arrays.copyOf(buckets, size);
    }

    public LogHistogram setBuckets(double[] buckets) {
        return setBuckets(buckets, buckets.length);
    }

    public LogHistogram setBuckets(double[] buckets, int length) {
        for (int index = 0; index < length; index++) {
            double value = buckets[index];
            if (Double.isNaN(value) || Double.compare(value, 0) < 0) {
                throw new IllegalArgumentException(
                    "Log histogram bucket can't contains nan or negative value, but specified: "
                        + Arrays.toString(buckets)
                );
            }
            this.buckets[index] = value;
        }
        this.size = length;
        return this;
    }

    public LogHistogram addBucket(double value) {
        if (Double.isNaN(value) || Double.compare(value, 0) < 0) {
            throw new IllegalArgumentException("Log histogram bucket can't contains nan or negative value, but specified: " + value);
        }

        this.buckets[size++] = value;
        return this;
    }

    public LogHistogram truncateBuckets() {
        if (size != 0) {
            int from = 0;
            while (from < size && buckets[from] == 0) {
                from++;
            }

            //buckets with only zeros
            if (from >= size) {
                size = 0;
                startPower = 0;
                return this;
            }

            int to = size;
            while (to > 0 && buckets[to - 1] == 0) {
                to--;
            }

            if (from != 0 || to != size) {
                System.arraycopy(buckets, from, buckets, 0, to - from);
                size = to - from;
                startPower += from;
            }
        }

        return this;
    }

    public LogHistogram build() {
        return truncateBuckets();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LogHistogram histogram = (LogHistogram) o;

        if (countZero != histogram.countZero) return false;
        if (startPower != histogram.startPower) return false;
        if (maxBucketsSize != histogram.maxBucketsSize) return false;
        if (Double.compare(histogram.base, base) != 0) return false;
        return Arrays.equals(buckets, 0, size, histogram.buckets, 0, histogram.size);
    }

    @Override
    public int hashCode() {
        int result;
        long temp;
        result = Arrays.hashCode(buckets);
        result = 31 * result + Long.hashCode(countZero);
        result = 31 * result + startPower;
        result = 31 * result + maxBucketsSize;
        temp = Double.doubleToLongBits(base);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "LogHistogram{" +
            "buckets=" + Arrays.toString(Arrays.copyOf(buckets, size)) +
            ", countZero=" + countZero +
            ", startPower=" + startPower +
            ", maxBucketsSize=" + maxBucketsSize +
            ", base=" + base +
            '}';
    }
}
