package ru.yandex.solomon.model.type;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.histogram.Arrays;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.point.column.HistogramColumn;

/**
 * @author Vladimir Gordiychuk
 */
public class MutableHistogram {
    private double[] bounds;
    private long[] buckets;
    private long denom;

    public MutableHistogram() {
        reset();
    }

    public void reset() {
        bounds = new double[0];
        buckets = new long[0];
    }

    public void addHistogram(Histogram snapshot) {
        if (isNullOrEmpty(snapshot)) {
            return;
        }
        changeDenom(snapshot.getDenom());
        if (!boundsEquals(snapshot)) {
            changeBounds(snapshot);
            return;
        }

        for (int index = 0; index < snapshot.count(); index++) {
            buckets[index] += snapshot.value(index);
        }
    }

    public Histogram snapshot() {
        return snapshot(null);
    }

    public Histogram snapshot(@Nullable Histogram histogram) {
        return Histogram.orNew(histogram)
            .copyFrom(bounds, buckets)
            .setDenom(denom);
    }

    private void changeDenom(long denom) {
        if (this.denom == denom) {
            return;
        }

        double multiplier = HistogramColumn.denomMultiplier(this.denom, denom);
        for (int index = 0; index < buckets.length; index++) {
            buckets[index] *= multiplier;
        }
        this.denom = denom;
    }

    private void changeBounds(Histogram snapshot) {
        double[] prevBounds = bounds;
        long[] prevBuckets = buckets;

        bounds = new double[snapshot.count()];
        buckets = new long[snapshot.count()];

        for (int index = 0; index < snapshot.count(); index++) {
            bounds[index] = snapshot.upperBound(index);
            buckets[index] = snapshot.value(index);
        }

        for (int index = 0; index < prevBounds.length; index++) {
            double upperBound = prevBounds[index];
            long count = prevBuckets[index];

            int bound = Arrays.lowerBound(bounds, upperBound);
            if (buckets.length == bound) {
                addInfBound();
                bound = buckets.length - 1;
            }

            buckets[bound] += count;
        }
    }

    private void addInfBound() {
        if (Double.compare(bounds[bounds.length - 1], Histograms.INF_BOUND) >= 0) {
            return;
        }

        if (bounds.length != Histograms.MAX_BUCKETS_COUNT) {
            bounds = java.util.Arrays.copyOf(bounds, bounds.length + 1);
            bounds[bounds.length - 1] = Histograms.INF_BOUND;
            buckets = java.util.Arrays.copyOf(buckets, buckets.length + 1);
            return;
        }

        bounds[bounds.length - 1] = Histograms.INF_BOUND;
    }

    private boolean boundsEquals(Histogram histogram) {
        if (bounds.length != histogram.count()) {
            return false;
        }

        for (int index = 0; index < bounds.length; index++) {
            if (Double.compare(bounds[index], histogram.upperBound(index)) != 0) {
                return false;
            }
        }

        return true;
    }

    private static boolean isNullOrEmpty(Histogram histogram) {
        return histogram == null || histogram.count() == 0;
    }
}
