package ru.yandex.solomon.model.type;

import java.util.Arrays;
import java.util.stream.DoubleStream;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MutableLogHistogram {
    private final double[] buckets = new double[LogHistogram.LIMIT_MAX_BUCKETS_SIZE];
    private long countZero;
    private int startPower;
    private int maxBucketsSize;
    private double base;
    private int size;

    public MutableLogHistogram() {
        reset();
    }

    private void copyState(LogHistogram initState) {
        reset();
        for (int index = 0; index < initState.countBucket(); index++) {
            this.buckets[index] = initState.getBucketValue(index);
        }
        this.size += initState.countBucket();
        this.countZero = initState.getCountZero();
        this.startPower = initState.getStartPower();
        this.maxBucketsSize = initState.getMaxBucketsSize();
        this.base = initState.getBase();
    }

    public MutableLogHistogram addHistogram(LogHistogram histogram) {
        if (size == 0 && countZero == 0) {
            copyState(histogram);
            return this;
        }

        // Overload previous merge result, in case when new histogram have
        // different left/right bounds, it's necessary because left and right bounds
        // contains inf sum, that can not be recalculate to correct bucket when
        // buckets size or weight changes, for example
        // first = buckets=[1, 2, 3, 4], countZero=0, startPower=-3, maxBucketsSize=5
        // second = buckets=[1, 2, 3, 4], countZero=0, startPower=3, maxBucketsSize=5
        // mergeOnSmall = merge(first, second)
        // When max bucket size equal to 5, we have left bounds overload and as a result sum points:
        // buckets=[10.0, 1.0, 2.0, 3.0, 4.0], countZero=0, startPower=2, maxBucketsSize=5
        // When max bucket size equal to 10, we doesn't have overload.
        // buckets=[1.0, 2.0, 3.0, 4.0, 0.0, 0.0, 1.0, 2.0, 3.0, 4.0], countZero=0, startPower=-3, maxBucketsSize=10
        if (histogram.getMaxBucketsSize() != maxBucketsSize) {
            copyState(histogram);
            return this;
        }

        actualizeBucketsToNewBase(histogram.getBase());
        mergeHistogram(histogram);
        return this;
    }

    private void actualizeBucketsToNewBase(double newBase) {
        if (Double.compare(base, newBase) == 0) {
            return;
        }

        int previousStartPower = startPower;
        startPower = 0;
        double[] previousBuckets = Arrays.copyOf(buckets, size);
        Arrays.fill(buckets, 0.0);

        double previousBase = base;
        base = newBase;

        for (int index = 0; index < previousBuckets.length; index++) {
            double countInBucket = previousBuckets[index];
            if (Double.compare(countInBucket, 0d) == 0) {
                continue;
            }

            int bucketIndex = index + previousStartPower;
            double bucketBound = Math.pow(previousBase, bucketIndex);
            int indexInNewBucket = changeBoundIfNecessary(estimateBucketIndexForValue(bucketBound));

            buckets[indexInNewBucket] += countInBucket;
        }
    }

    private void mergeHistogram(LogHistogram histogram) {
        countZero += histogram.getCountZero();

        int fromInclusive = histogram.getStartPower() - this.startPower;
        int toExclusive = fromInclusive + histogram.countBucket();

        int firstIndex = extendBounds(fromInclusive, toExclusive);
        int toMerge = Math.max(-firstIndex, 0);  // if negative value, we should merge exceeded buckets

        if (toMerge > 0) {
            buckets[0] += IntStream.range(0, Math.min(toMerge, histogram.countBucket())).mapToDouble(histogram::getBucketValue).sum();
            firstIndex = 0;
        }

        int bucketIndex = firstIndex;
        for (int index = toMerge; index < histogram.countBucket(); index++) {
            buckets[bucketIndex++] += histogram.getBucketValue(index);
        }
    }

    public void addValues(double... values) {
        for (double value : values) {
            addValue(value);
        }
    }

    public void addValue(double value) {
        addValue(value, 1);
    }

    public void addValue(double value, long count) {
        if (Double.isNaN(value)) {
            return;
        }

        if (Double.isInfinite(value)) {
            throw new IllegalArgumentException("Not able add " + value + " because infinite and nan value restricted, log histogram: " + this);
        } else if (Double.compare(value, 0d) <= 0) {
            countZero += count;
            return;
        }

        int index = changeBoundIfNecessary(estimateBucketIndexForValue(value));
        buckets[index] += count;
    }

    private int changeBoundIfNecessary(int requiredIndex) {
        if (requiredIndex >= maxBucketsSize) {
            return extendUp(requiredIndex + 1) - 1;
        } else if (requiredIndex < 0) {
            extendDown(requiredIndex);
            return 0;
        }

        size = Math.max(requiredIndex + 1, size);
        return requiredIndex;
    }

    private int estimateBucketIndexForValue(double value) {
        return  (int) (Math.floor(Math.log(value) / Math.log(this.base)) - startPower);
    }

    private int extendBounds(int fromInclusive, int toExclusive) {
        int realStartIndex = fromInclusive;
        if (toExclusive > maxBucketsSize) {
            int newLength = extendUp(toExclusive);
            size = newLength;
            int shift = newLength - toExclusive;
            realStartIndex += shift;
        } else {
            size = Math.max(size, toExclusive);
        }

        if (realStartIndex < 1) {
            realStartIndex = extendDown(realStartIndex);
        }

        return realStartIndex;
    }

    private int extendDown(int fromIndex) {
        int addSize = Math.min((maxBucketsSize - size), Math.abs(fromIndex));
        if (addSize > 0) {
            System.arraycopy(buckets, 0, buckets, addSize, size);
            Arrays.fill(buckets, 0, addSize, 0.0);
            startPower -= addSize;
            this.size += addSize;
        }

        return fromIndex + addSize;
    }

    private int extendUp(int expectLength) {
        if (expectLength > maxBucketsSize) {
            int countItemToSum = expectLength - maxBucketsSize;
            buckets[0] = DoubleStream.of(buckets).limit(countItemToSum + 1).sum();
            if (countItemToSum < buckets.length) {
                System.arraycopy(
                    buckets,
                    countItemToSum + 1,
                    buckets,
                    1,
                    buckets.length - countItemToSum - 1
                );
            }

            startPower += countItemToSum;
        }

        return Math.min(expectLength, maxBucketsSize);
    }

    public LogHistogram toImmutable() {
        return LogHistogram.newBuilder()
            .setBuckets(buckets, size)
            .setStartPower(startPower)
            .setCountZero(countZero)
            .setMaxBucketsSize(maxBucketsSize)
            .setBase(base)
            .build();
    }

    public LogHistogram build(LogHistogram value) {
        value.reset();
        return value.setBuckets(buckets, size)
            .setStartPower(startPower)
            .setCountZero(countZero)
            .setMaxBucketsSize(maxBucketsSize)
            .setBase(base)
            .build();
    }

    public void reset() {
        Arrays.fill(this.buckets, 0.0);
        this.countZero = 0;
        this.startPower = 0;
        this.maxBucketsSize = LogHistogram.DEFAULT_MAX_BUCKET_SIZE;
        this.base = LogHistogram.DEFAULT_BASE;
        this.size = 0;
    }
}
