package ru.yandex.solomon.model.type;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;

/**
 * Copy-past from {@link java.util.DoubleSummaryStatistics} with support reset to be able
 * reuse same object for multiple aggregations.
 *
 * @author Vladimir Gordiychuk
 * @see java.util.DoubleSummaryStatistics
 */
public class MutableSummaryDouble {
    private long count;
    private double sum;
    private double sumCompensation; // Low order bits of sum
    private double simpleSum; // Used to compute right sum for non-finite inputs
    private double min = Double.POSITIVE_INFINITY;
    private double max = Double.NEGATIVE_INFINITY;
    private double last;

    public void reset() {
        count = 0;
        sum = 0;
        sumCompensation = 0;
        simpleSum = 0;
        min = Double.POSITIVE_INFINITY;
        max = Double.NEGATIVE_INFINITY;
        last = 0;
    }

    public void addSummary(SummaryDoubleSnapshot snapshot) {
        count += snapshot.getCount();
        simpleSum += snapshot.getSum();
        sumWithCompensation(snapshot.getSum());
        min = Math.min(min, snapshot.getMin());
        max = Math.max(max, snapshot.getMax());
        last = snapshot.getLast();
    }

    public final long getCount() {
        return count;
    }

    public final double getSum() {
        // Better error bounds to add both terms as the final sum
        double tmp = sum + sumCompensation;
        if (Double.isNaN(tmp) && Double.isInfinite(simpleSum))
            // If the compensated sum is spuriously NaN from
            // accumulating one or more same-signed infinite values,
            // return the correctly-signed infinity stored in
            // simpleSum.
            return simpleSum;
        else
            return tmp;
    }

    public final double getMin() {
        return min;
    }

    public final double getMax() {
        return max;
    }

    public final double getAverage() {
        return getCount() > 0 ? getSum() / getCount() : 0.0d;
    }

    public final double getLast() {
        return last;
    }

    public SummaryDoubleSnapshot snapshot() {
        return new ImmutableSummaryDoubleSnapshot(getCount(), getSum(), getMin(), getMax(), getLast());
    }

    public SummaryDoubleSnapshot snapshot(@Nullable SummaryDoubleSnapshot prev) {
        return SummaryDouble.orNew(prev)
            .setCount(getCount())
            .setSum(getSum())
            .setMin(getMin())
            .setMax(getMax())
            .setLast(getLast());
    }

    /**
     * Incorporate a new double value using Kahan summation /
     * compensated summation.
     */
    private void sumWithCompensation(double value) {
        double tmp = value - sumCompensation;
        double velvel = sum + tmp; // Little wolf of rounding error
        sumCompensation = (velvel - sum) - tmp;
        sum = velvel;
    }
}
