package ru.yandex.solomon.model.type;

import javax.annotation.Nullable;

import io.netty.util.Recycler;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryDouble implements SummaryDoubleSnapshot {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SummaryDouble.class);
    private static Recycler<SummaryDouble> RECYCLE = new Recycler<>() {
        @Override
        protected SummaryDouble newObject(Handle<SummaryDouble> handle) {
            return new SummaryDouble(handle);
        }
    };

    private final Recycler.Handle<SummaryDouble> recycleHandler;
    private long count;
    private double sum;
    private double min;
    private double max;
    private double last;

    private SummaryDouble(Recycler.Handle<SummaryDouble> recycleHandler) {
        this.recycleHandler = recycleHandler;
    }

    public static SummaryDouble newInstance() {
        return RECYCLE.get();
    }

    public static SummaryDouble newInstance(long count, double sum, double min, double max) {
        return newInstance(count, sum, min, max, 0);
    }

    public static SummaryDouble newInstance(long count, double sum, double min, double max, double last) {
        SummaryDouble summary = newInstance();
        summary.setCount(count);
        summary.setSum(sum);
        summary.setMin(min);
        summary.setMax(max);
        summary.setLast(last);
        return summary;
    }

    public static SummaryDouble copyOf(SummaryDoubleSnapshot snapshot) {
        return newInstance().copyFrom(snapshot);
    }

    public static SummaryDouble orNew(@Nullable SummaryDoubleSnapshot snapshot) {
        if (snapshot instanceof SummaryDouble) {
            var result = (SummaryDouble) snapshot;
            result.reset();
            return result;
        }

        return newInstance();
    }

    public static void recycle(@Nullable SummaryDoubleSnapshot snapshot) {
        if (snapshot == null) {
            return;
        }

        if (snapshot instanceof SummaryDouble) {
            ((SummaryDouble) snapshot).recycle();
        }
    }

    public SummaryDouble copyFrom(SummaryDoubleSnapshot snapshot) {
        count = snapshot.getCount();
        sum = snapshot.getSum();
        min = snapshot.getMin();
        max = snapshot.getMax();
        last = snapshot.getLast();
        return this;
    }

    public void reset() {
        count = 0;
        sum = 0;
        min = Double.POSITIVE_INFINITY;
        max = Double.NEGATIVE_INFINITY;
        last = 0;
    }

    @Override
    public long getCount() {
        return count;
    }

    public SummaryDouble setCount(long count) {
        this.count = count;
        return this;
    }

    @Override
    public double getSum() {
        return sum;
    }

    public SummaryDouble setSum(double sum) {
        this.sum = sum;
        return this;
    }

    @Override
    public double getMin() {
        return min;
    }

    public SummaryDouble setMin(double min) {
        this.min = min;
        return this;
    }

    @Override
    public double getMax() {
        return max;
    }

    public SummaryDouble setMax(double max) {
        this.max = max;
        return this;
    }

    @Override
    public double getLast() {
        return last;
    }

    public SummaryDouble setLast(double last) {
        this.last = last;
        return this;
    }

    public void recycle() {
        reset();
        recycleHandler.recycle(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof SummaryDoubleSnapshot)) {
            return false;
        }
        SummaryDoubleSnapshot that = (SummaryDoubleSnapshot) o;

        if (count != that.getCount()) return false;
        if (Double.compare(that.getSum(), sum) != 0) return false;
        if (Double.compare(that.getMin(), min) != 0) return false;
        if (Double.compare(that.getMax(), max) != 0) return false;
        return Double.compare(that.getLast(), last) == 0;
    }

    @Override
    public int hashCode() {
        int result;
        long temp;
        result = (int) (count ^ (count >>> 32));
        temp = Double.doubleToLongBits(sum);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(min);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(max);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(last);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        sb.append("count: ").append(getCount());
        sb.append(", sum: ").append(getSum());
        if (getMin() != ImmutableSummaryDoubleSnapshot.EMPTY.getMin()) {
            sb.append(", min: ").append(getMin());
        }

        if (getMax() != ImmutableSummaryDoubleSnapshot.EMPTY.getMax()) {
            sb.append(", max: ").append(getMax());
        }
        if (getLast() != 0) {
            sb.append(", last: ").append(getLast());
        }
        sb.append('}');
        return sb.toString();
    }
}
