package ru.yandex.solomon.model.type;

import javax.annotation.Nullable;

import io.netty.util.Recycler;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryInt64 implements SummaryInt64Snapshot {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SummaryInt64.class);
    private static Recycler<SummaryInt64> RECYCLE = new Recycler<>() {
        @Override
        protected SummaryInt64 newObject(Handle<SummaryInt64> handle) {
            return new SummaryInt64(handle);
        }
    };

    private final Recycler.Handle<SummaryInt64> recycleHandler;
    private long count;
    private long sum;
    private long min;
    private long max;
    private long last;

    private SummaryInt64(Recycler.Handle<SummaryInt64> recycleHandler) {
        this.recycleHandler = recycleHandler;
    }

    public static SummaryInt64 newInstance() {
        return RECYCLE.get();
    }

    public static SummaryInt64 newInstance(long count, long sum, long min, long max) {
        return newInstance(count, sum, min, max, 0);
    }

    public static SummaryInt64 newInstance(long count, long sum, long min, long max, long last) {
        SummaryInt64 summary = newInstance();
        summary.setCount(count);
        summary.setSum(sum);
        summary.setMin(min);
        summary.setMax(max);
        summary.setLast(last);
        return summary;
    }

    public static SummaryInt64 copyOf(SummaryInt64Snapshot snapshot) {
        return newInstance().copyFrom(snapshot);
    }

    public static SummaryInt64 orNew(@Nullable SummaryInt64Snapshot snapshot) {
        if (snapshot instanceof SummaryInt64) {
            var result = (SummaryInt64) snapshot;
            result.reset();
            return result;
        }

        return newInstance();
    }

    public static void recycle(@Nullable SummaryInt64Snapshot snapshot) {
        if (snapshot == null) {
            return;
        }

        if (snapshot instanceof SummaryInt64) {
            ((SummaryInt64) snapshot).recycle();
        }
    }

    public SummaryInt64 copyFrom(SummaryInt64Snapshot snapshot) {
        count = snapshot.getCount();
        sum = snapshot.getSum();
        min = snapshot.getMin();
        max = snapshot.getMax();
        last = snapshot.getLast();
        return this;
    }

    public void reset() {
        count = 0;
        sum = 0;
        min = Long.MAX_VALUE;
        max = Long.MIN_VALUE;
        last = 0;
    }

    @Override
    public long getCount() {
        return count;
    }

    public SummaryInt64 setCount(long count) {
        this.count = count;
        return this;
    }

    @Override
    public long getSum() {
        return sum;
    }

    public SummaryInt64 setSum(long sum) {
        this.sum = sum;
        return this;
    }

    @Override
    public long getMin() {
        return min;
    }

    public SummaryInt64 setMin(long min) {
        this.min = min;
        return this;
    }

    @Override
    public long getMax() {
        return max;
    }

    public SummaryInt64 setMax(long max) {
        this.max = max;
        return this;
    }

    @Override
    public long getLast() {
        return last;
    }

    public SummaryInt64 setLast(long last) {
        this.last = last;
        return this;
    }

    public void recycle() {
        reset();
        recycleHandler.recycle(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof SummaryInt64Snapshot)) {
            return false;
        }
        SummaryInt64Snapshot that = (SummaryInt64Snapshot) o;

        if (count != that.getCount()) return false;
        if (that.getSum() != sum) return false;
        if (that.getMin() != min) return false;
        if (that.getMax() != max) return false;
        return that.getLast() == last;
    }

    @Override
    public int hashCode() {
        int result = (int) (count ^ (count >>> 32));
        result = 31 * result + (int) (sum ^ (sum >>> 32));
        result = 31 * result + (int) (min ^ (min >>> 32));
        result = 31 * result + (int) (max ^ (max >>> 32));
        result = 31 * result + (int) (last ^ (last >>> 32));
        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        sb.append("count: ").append(getCount());
        sb.append(", sum: ").append(getSum());
        if (getMin() != ImmutableSummaryInt64Snapshot.EMPTY.getMin()) {
            sb.append(", min: ").append(getMin());
        }

        if (getMax() != ImmutableSummaryInt64Snapshot.EMPTY.getMax()) {
            sb.append(", max: ").append(getMax());
        }
        if (getLast() != 0) {
            sb.append(", last: ").append(getLast());
        }
        sb.append('}');
        return sb.toString();
    }
}
