package ru.yandex.solomon.model.type.ugram;

import java.util.Iterator;
import java.util.List;

import javax.annotation.Nullable;

import io.netty.util.Recycler;
import io.netty.util.internal.RecyclableArrayList;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.type.Histogram;

import static ru.yandex.solomon.model.point.column.HistogramColumn.isNullOrEmpty;

/**
 * @author Vladimir Gordiychuk
 */
public class Ugram {
    private static final Recycler<Ugram> RECYCLER = new Recycler<>() {
        protected Ugram newObject(Handle<Ugram> handle) {
            return new Ugram(handle);
        }
    };

    private final Recycler.Handle<Ugram> handle;

    private List<Bucket> buckets;

    public Ugram(Recycler.Handle<Ugram> handle) {
        this.handle = handle;
    }

    public static Ugram create() {
        var ugram = RECYCLER.get();
        ugram.buckets = (List) RecyclableArrayList.newInstance(Histograms.MAX_BUCKETS_COUNT);
        return ugram;
    }

    public void reset() {
        for (Bucket bucket : buckets) {
            bucket.recycle();
        }
        buckets.clear();
    }

    public void recycle() {
        reset();
        List list = buckets;
        ((RecyclableArrayList) list).recycle();
        handle.recycle(this);
    }

    // https://a.yandex-team.ru/arc/trunk/arcadia/infra/yasm/zoom/components/hgram/ugram/ugram.cpp?rev=4319857#L55
    public void merge(Histogram snapshot) {
        if (isNullOrEmpty(snapshot)) {
            return;
        }

        if (buckets.isEmpty()) {
            var it = new SnapshotBuckets(snapshot);
            while (it.hasNext()) {
                buckets.add(it.next());
            }
            return;
        }

        var copy = buckets;
        try {
            buckets = (List) RecyclableArrayList.newInstance(buckets.size() + snapshot.count());
            sortAndMerge(copy, snapshot);
        } finally {
            List list = copy;
            ((RecyclableArrayList) list).recycle();
        }
    }

    private void sortAndMerge(List<Bucket> left, Histogram right) {
        int leftIdx = 0;
        int rightIdx = 0;
        double prevRightUpper = 0;
        var rightBucket = Bucket.create();
        boolean cleanRightBucket = true;
        rightLoop:
        while (rightIdx < right.count()) {
            rightBucket.lowerBound = prevRightUpper;
            rightBucket.upperBound = right.upperBound(rightIdx);
            rightBucket.weight = right.value(rightIdx);
            prevRightUpper = rightBucket.upperBound;

            while (leftIdx < left.size()) {
                var leftBucket = left.get(leftIdx);
                int compare = leftBucket.compareTo(rightBucket);
                if (compare == 0) {
                    // fast path, bounds are equal
                    leftBucket.weight += rightBucket.weight;
                    mergeInternal(leftBucket);
                    rightIdx++;
                    leftIdx++;
                    continue rightLoop;
                } else if (compare < 0) {
                    mergeInternal(leftBucket);
                    leftIdx++;
                } else {
                    mergeInternal(rightBucket);
                    rightIdx++;
                    rightBucket = Bucket.create();
                    continue rightLoop;
                }
            }

            cleanRightBucket = false;
            mergeInternal(rightBucket);
            for (rightIdx++; rightIdx < right.count(); rightIdx++) {
                rightBucket = Bucket.create();
                rightBucket.lowerBound = prevRightUpper;
                rightBucket.upperBound = right.upperBound(rightIdx);
                rightBucket.weight = right.value(rightIdx);
                prevRightUpper = rightBucket.upperBound;
                mergeInternal(rightBucket);
            }
        }

        if (cleanRightBucket) {
            rightBucket.recycle();
        }

        for (; leftIdx < left.size(); leftIdx++) {
            mergeInternal(left.get(leftIdx));
        }
    }

    private void mergeInternal(Bucket bucket) {
        int prevIdx = buckets.size();
        if (prevIdx == 0) {
            buckets.add(bucket);
            return;
        }

        Bucket prev = buckets.get(prevIdx - 1);
        if (prev.upperBound <= bucket.lowerBound) {
            buckets.add(bucket);
            return;
        }

        if (Double.compare(prev.upperBound, bucket.upperBound) == 0) {
            prev.weight += bucket.weight;
            bucket.recycle();
            return;
        }

        double prevWeight = prev.weight;
        double prevUpper = prev.upperBound;
        double prevSize = prevUpper - prev.lowerBound;
        double bucketSize = bucket.upperBound - bucket.lowerBound;

        if (prevUpper >= bucket.upperBound) {
            // nesting case
            // prev: Bucket{lowerBound=40.0, upperBound=50.0, weight=6.0}
            // next: Bucket{lowerBound=40.0, upperBound=45.0, weight=5.0}
            prev.upperBound = bucket.upperBound;
            prev.weight = bucket.weight + prevWeight * bucketSize / prevSize;

            bucket.weight = prevWeight * (prevUpper - bucket.upperBound) / prevSize;
            bucket.lowerBound = bucket.upperBound;
            bucket.upperBound = prevUpper;
            buckets.add(bucket);
        } else {
            // intersection case
            // prev: Bucket{lowerBound=10.0, upperBound=30.0, weight=20.0}
            // next: Bucket{lowerBound=10.0, upperBound=50.0, weight=40.0}
            prev.weight = prevWeight * (prevUpper - bucket.lowerBound) / prevSize + bucket.weight * (prevUpper - bucket.lowerBound) / bucketSize;

            bucket.lowerBound = prevUpper;
            bucket.weight = bucket.weight * (bucket.upperBound - prevUpper) / bucketSize;
            buckets.add(bucket);
        }
    }

    public Histogram snapshot() {
        return snapshot(null);
    }

    public Histogram snapshot(@Nullable Histogram hist) {
        Compress.compress(buckets);
        var result = Histogram.orNew(hist);
        for (int index = 0; index < buckets.size(); index++) {
            var bucket = (Bucket) buckets.get(index);
            result.setUpperBound(index, bucket.upperBound);
            result.setBucketValue(index, Math.round(bucket.weight));
        }
        return result;
    }

    private static class SnapshotBuckets implements Iterator<Bucket> {
        private Histogram snapshot;
        private int index;
        private double prevUpperBound;

        SnapshotBuckets(Histogram snapshot) {
            this.snapshot = snapshot;
        }

        @Override
        public boolean hasNext() {
            return index < snapshot.count();
        }

        @Override
        public Bucket next() {
            return next(Bucket.create());
        }

        public Bucket next(Bucket bucket) {
            bucket.lowerBound = prevUpperBound;
            bucket.upperBound = snapshot.upperBound(index);
            bucket.weight = snapshot.value(index);
            prevUpperBound = bucket.upperBound;
            index++;
            return bucket;
        }
    }


}
