package ru.yandex.solomon.util.mh;

import java.lang.invoke.MethodHandle;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.infra.Blackhole;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

/**
 * @author Stepan Koltsov
 */
@State(Scope.Thread)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
@Warmup(iterations = 5)
@Measurement(iterations = 6)
@BenchmarkMode(Mode.AverageTime)
@Fork(2)
public class MultiArrayResizerBenchmark {

    public static class DataForResizer {
        private int[] ints = new int[0];
        private long[] longs = new long[0];
        private String[] strings = new String[0];

        @ArrayListSize
        private int size = 0;
    }

    private static class MultiArrayResizerDirect implements MultiArrayResizer<DataForResizer> {

        @Override
        public void resize(DataForResizer arrays, int newSize) {
            arrays.ints = Arrays.copyOf(arrays.ints, newSize);
            arrays.longs = Arrays.copyOf(arrays.longs, newSize);
            arrays.strings = Arrays.copyOf(arrays.strings, newSize);
        }

        @Override
        public int capacity(DataForResizer arrays) {
            return arrays.ints.length;
        }

        @Override
        public int size(DataForResizer arrays) {
            return arrays.size;
        }

        @Override
        public void reserveAdditional(DataForResizer arrays, int additional) {
            reserveAdditionalSimple(arrays, additional);
        }
    }

    private static final MultiArrayResizerRefl<DataForResizer> reflResizer = new MultiArrayResizerRefl<>(DataForResizer.class);
    private static final MultiArrayResizerMh<DataForResizer> mhResizer = new MultiArrayResizerMh<>(DataForResizer.class);
    private static final MultiArrayResizerDirect directResizer = new MultiArrayResizerDirect();
    private static final MethodHandle mh = mhResizer.getReserveAdditionalMhAParam();

    @Benchmark
    public void reflResizer(Blackhole blackhole) {
        DataForResizer data = new DataForResizer();
        blackhole.consume(data);
        for (int i = 0; i < 20; ++i) {
            reflResizer.reserveAdditional(data, 1);
            data.size += 1;
            blackhole.consume(data);
        }
    }

    @Benchmark
    public void directResizer(Blackhole blackhole) {
        DataForResizer data = new DataForResizer();
        blackhole.consume(data);
        for (int i = 0; i < 20; ++i) {
            directResizer.reserveAdditional(data, 1);
            data.size += 1;
            blackhole.consume(data);
        }
    }

    @Benchmark
    public void mhResizer(Blackhole blackhole) {
        DataForResizer data = new DataForResizer();
        blackhole.consume(data);
        for (int i = 0; i < 20; ++i) {
            mhResizer.reserveAdditional(data, 1);
            data.size += 1;
            blackhole.consume(data);
        }
    }

    @Benchmark
    public void mhResizerSimple(Blackhole blackhole) {
        DataForResizer data = new DataForResizer();
        blackhole.consume(data);
        for (int i = 0; i < 20; ++i) {
            mhResizer.reserveAdditionalSimple(data, 1);
            data.size += 1;
            blackhole.consume(data);
        }
    }

    @Benchmark
    public void mh(Blackhole blackhole) {
        DataForResizer data = new DataForResizer();
        blackhole.consume(data);
        for (int i = 0; i < 20; ++i) {
            try {
                mh.invokeExact(data, 1);
            } catch (Throwable e) {
                throw new RuntimeException(e);
            }
            data.size += 1;
            blackhole.consume(data);
        }
    }

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(MultiArrayResizerBenchmark.class.getSimpleName())
                .build();
        new Runner(opt).run();
    }
}
