package ru.yandex.solomon.util;

import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.slf4j.Logger;

import ru.yandex.solomon.util.exception.RpcException;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Stepan Koltsov
 */
public class LoggerUtils {
    private static final long FORCE_LOG_DURATION_MILLIS = 200;

    public static <T> T logOperation(Logger logger, String operation, Supplier<T> impl) {
        logStart(logger, operation);
        long start = System.currentTimeMillis();
        try {
            T r = impl.get();
            logCompletionIntermediate(logger, operation, null, start);
            return r;
        } catch (Throwable e) {
            long d = System.currentTimeMillis() - start;
            logCompletionIntermediate(logger, operation, e, d);
            throw ru.yandex.misc.ExceptionUtils.throwException(e);
        }
    }

    public static void logStart(Logger logger, String operation) {
        if (logger.isDebugEnabled()) {
            logger.debug(operation + "...");
        }
    }

    public static void logCompletionIntermediate(Logger logger, String operation, @Nullable Throwable e, long start) {
        logCompletionIntermediate(logger, operation, e != null, start);
    }

    public static void logCompletionIntermediate(Logger logger, String operation, boolean error, long start) {
        if (isLongDuration(start) || logger.isDebugEnabled()) {
            logger.debug(operation
                    + " " + DurationUtils.millisToSecondsStringToNow(start)
                    // not logging error, because it should be logged by caller
                    + (error ? " error" : ""));
        }
    }

    public static void logCompletionFinal(Logger logger, String operation, @Nullable Throwable e, long start) {
        if (e != null) {
            boolean log = true;
            Throwable t = e;
            while (t != null) {
                if ((t instanceof RpcException)) {
                    log = ((RpcException) t).logOnRpcServer();
                    break;
                }
                t = t.getCause();
            }
            if (log) {
                logger.error(operation + " " + DurationUtils.millisToSecondsStringToNow(start), e);
            }
        } else if (isLongDuration(start) || logger.isDebugEnabled()) {
            logger.debug(operation + " " + DurationUtils.millisToSecondsStringToNow(start));
        }
    }

    private static boolean isLongDuration(long startMillis) {
        long dtMillis = System.currentTimeMillis() - startMillis;
        return dtMillis >= FORCE_LOG_DURATION_MILLIS;
    }

}
