package ru.yandex.solomon.util;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.LabelsBuilder;

/**
 * Helper to analyze metric name in labels and order to execute one of:
 * 1) Do nothing.
 * 2) Add metric name label.
 * 3) Throw exception.
 * Supports old (labels only) and new (metricName, labels) formats.
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class MetricNameHelper {

    /**
     * Convert (metricName, labels) to labels in old format
     */
    public static void convertToOldFormat(
        LabelsBuilder builder,
        String metricNameLabel,
        String metricName)
    {
        boolean canAdd = canAddMetricNameLabel(builder, metricNameLabel, metricName);
        if (canAdd) {
            builder.add(metricNameLabel, metricName);
        }
    }

    /**
     * Check that you can add metric name label to labels to convert them to old labels format
     *
     * @param builder labels builder
     * @param metricNameLabel required metric name label in shard, e.g. "metric"
     * @param metricName metric name in case of new data format
     *                   or empty in case of old data format
     * @return true if you can add metric name label to labels, false otherwise
     * @throws InvalidMetricNameException in case of invalid metric name
     */
    public static boolean canAddMetricNameLabel(
        LabelsBuilder builder,
        String metricNameLabel,
        String metricName) throws InvalidMetricNameException
    {
        boolean newDataFormat = !metricName.isEmpty();
        boolean hasMetricNameLabel = !metricNameLabel.isEmpty();

        if (hasMetricNameLabel) {
            return analyzeNameForNewShard(builder, metricNameLabel, newDataFormat);
        }

        return analyzeMetricNameForOldShard(newDataFormat);
    }

    private static boolean analyzeMetricNameForOldShard(boolean newDataFormat) {
        if (newDataFormat) {
            throw new InvalidMetricNameException("unexpected new data format for shard without required metric name label");
        }

        return false;
    }

    private static boolean analyzeNameForNewShard(
        LabelsBuilder builder,
        String nameLabel,
        boolean newDataFormat)
    {
        if (newDataFormat) {
            if (builder.hasKey(nameLabel)) {
                String message =
                    "labels " + builder.build() + " already contain alias label for metric name: "
                        + nameLabel;
                throw new InvalidMetricNameException(message);
            }

            return true;
        } else {
            if (!builder.hasKey(nameLabel)) {
                String message =
                    "labels " + builder.build() + "doesn't contain required metric name label: "
                        + nameLabel;
                throw new InvalidMetricNameException(message);
            }

            return false;
        }
    }
}
