package ru.yandex.solomon.util;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadFactory;

import io.netty.channel.EventLoopGroup;
import io.netty.channel.ServerChannel;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.epoll.EpollDomainSocketChannel;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.kqueue.KQueue;
import io.netty.channel.kqueue.KQueueDatagramChannel;
import io.netty.channel.kqueue.KQueueDomainSocketChannel;
import io.netty.channel.kqueue.KQueueEventLoopGroup;
import io.netty.channel.kqueue.KQueueServerSocketChannel;
import io.netty.channel.kqueue.KQueueSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.channel.unix.DomainSocketChannel;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;


/**
 * @author Sergey Polovko
 */
public final class NettyUtils {
    private NettyUtils() {}

    public static Class<? extends SocketChannel> clientChannelType() {
        if (Epoll.isAvailable()) {
            return EpollSocketChannel.class;
        } else if (KQueue.isAvailable()) {
            return KQueueSocketChannel.class;
        }
        return NioSocketChannel.class;
    }

    public static Class<? extends ServerChannel> serverChannelType() {
        if (Epoll.isAvailable()) {
            return EpollServerSocketChannel.class;
        } else if (KQueue.isAvailable()) {
            return KQueueServerSocketChannel.class;
        }
        return NioServerSocketChannel.class;
    }

    public static Class<? extends SocketChannel> clientChannelTypeForEventLoop(EventLoopGroup eventLoopGroup) {
        if (eventLoopGroup instanceof EpollEventLoopGroup) {
            return EpollSocketChannel.class;
        } else if (eventLoopGroup instanceof KQueueEventLoopGroup) {
            return KQueueSocketChannel.class;
        } else if (eventLoopGroup instanceof NioEventLoopGroup) {
            return NioSocketChannel.class;
        }
        String typeName = eventLoopGroup == null ? "null" : eventLoopGroup.getClass().getName();
        throw new IllegalArgumentException("unsupported event loop group type: " + typeName);
    }


    public static Class<? extends DomainSocketChannel> clientDomainSocketChannelForEventLoop(EventLoopGroup eventLoopGroup) {
        if (eventLoopGroup instanceof  EpollEventLoopGroup) {
            return EpollDomainSocketChannel.class;
        } else if (eventLoopGroup instanceof KQueueEventLoopGroup) {
            return KQueueDomainSocketChannel.class;
        }

        String typeName = eventLoopGroup == null ? "null" : eventLoopGroup.getClass().getName();
        throw new IllegalArgumentException("unsupported event loop group type: " + typeName);
    }

    public static Class<? extends DatagramChannel> clientDatagramChannelForEventLoop(EventLoopGroup eventLoopGroup) {
        if (eventLoopGroup instanceof  EpollEventLoopGroup) {
            return EpollDatagramChannel.class;
        } else if (eventLoopGroup instanceof KQueueEventLoopGroup) {
            return KQueueDatagramChannel.class;
        } else if (eventLoopGroup instanceof NioEventLoopGroup) {
            return NioDatagramChannel.class;
        }

        String typeName = eventLoopGroup == null ? "null" : eventLoopGroup.getClass().getName();
        throw new IllegalArgumentException("unsupported event loop group type: " + typeName);
    }

    public static EventLoopGroup createEventLoopGroup(String name, int nThreads) {
        ThreadFactory threadFactory = new DefaultThreadFactory(name);
        if (Epoll.isAvailable()) {
            return new EpollEventLoopGroup(nThreads, threadFactory);
        } else if (KQueue.isAvailable()) {
            return new KQueueEventLoopGroup(nThreads, threadFactory);
        }
        return new NioEventLoopGroup(nThreads, threadFactory);
    }

    public static <V> CompletableFuture<V> toCompletableFuture(Future<V> future) {
        if (future.isDone()) {
            if (future.isSuccess()) {
                return CompletableFuture.completedFuture(future.getNow());
            } else {
                return CompletableFuture.failedFuture(future.cause());
            }
        }

        var promise = new CompletableFuture<V>();
        future.addListener((GenericFutureListener<Future<V>>) f -> {
            if (f.isSuccess()) {
                promise.complete(f.getNow());
            } else {
                promise.completeExceptionally(f.cause());
            }
        });
        return promise;
    }
}
