package ru.yandex.solomon.util;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class Quoter {

    private static final Escaper SINGLE_QUOTE_ESCAPER = new Escaper("'", '\\');
    private static final Escaper DOUBLE_QUOTE_ESCAPER = new Escaper("\"", '\\');

    /**
     * Consume single-quoted, double-quoted or unquoted string.
     * Unquoted string will be consumed until whitespace or specified stop symbols.
     *
     * @param parser parser over source string
     * @param stopSymbols set of special stop symbols for unquoted string.
     *                    It may be delimiter or forbidden symbols.
     * @return consumed and unquoted string
     */
    public static String consumeQuotedString(
        ParserSupport parser,
        Set<Character> stopSymbols)
    {
        if (!parser.hasNext()) {
            return "";
        }

        char quote = parser.lookahead();

        if (quote == '\'' || quote == '"') {
            parser.consumeChar();
            Escaper escaper = quote == '\'' ? SINGLE_QUOTE_ESCAPER : DOUBLE_QUOTE_ESCAPER;
            String result = escaper.consumeUntil(parser, quote);
            if (!parser.hasNext()) {
                throw new RuntimeException("quote must be closed");
            }
            parser.consume(quote);
            return result;
        }

        return parser.consumeWhile(p -> !Character.isWhitespace(p) && !stopSymbols.contains(p));
    }

    /**
     * Quote text by double-quotes with escaping
     *
     * @param text text, e.g.: "smth"
     * @return quoted text, e.g.: "\"smth\""
     */
    public static String doubleQuote(String text) {
        return '"' + StringUtils.replace(text, "\"", "\\\"") + '"';
    }

    /**
     * Quote text by single-quotes with escaping
     *
     * @param text text, e.g.: 'smth'
     * @return quoted text, e.g.: '\'smth\''
     */
    public static String singleQuote(String text) {
        return '\'' + StringUtils.replace(text, "\'", "\\\'") + '\'';
    }
}
