package ru.yandex.solomon.util;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ServicePathEncoder {

    /**
     * Urlencode service path for fetcher requests
     *
     * @param path url-encoded or human-readable service path to urlencoded path
     * @return urlencoded service path
     * @throws URISyntaxException if path syntax is invalid
     * @throws UnsupportedEncodingException if URL encoder/decoder can't support current encoding
     * @throws IllegalArgumentException if it's impossible to decode path because it is invalid
     */
    public static String encode(String path)
        throws URISyntaxException, IllegalArgumentException, UnsupportedEncodingException {
        String decodedPath = urldecode(path);
        URI uri = new URI("http://localhost", decodedPath, null);

        String pathPart = Nullables.orEmpty(uri.getPath());
        String queryPart = Nullables.orEmpty(uri.getQuery());

        if (queryPart.isEmpty()) {
            return pathPart;
        }

        String[] pairs = StringUtils.split(queryPart, '&');

        StringBuilder stringBuilder = new StringBuilder();

        for (int i = 0; i < pairs.length; i++) {
            if (i > 0) {
                stringBuilder.append('&');
            }

            String pair = pairs[i];
            int eqIndex = pair.indexOf('=');

            if (eqIndex == 0) {
                throw new IllegalArgumentException("invalid query pair in url: " + pair);
            }
            if (eqIndex < 0) {
                stringBuilder.append(urlencode(pair));
            } else {
                String key = pair.substring(0, eqIndex);
                String value = pair.substring(eqIndex + 1);
                stringBuilder.append(urlencode(key)).append('=').append(urlencode(value));
            }
        }

        String encodedQuery = stringBuilder.toString();

        return pathPart + "?" + encodedQuery;
    }

    private static String urldecode(String path) throws UnsupportedEncodingException {
        return URLDecoder.decode(StringUtils.replace(path, "+", "%2b"), "UTF-8");
    }

    private static String urlencode(String s) throws UnsupportedEncodingException {
        return URLEncoder.encode(s, "UTF-8").replace("+", "%20");
    }
}
