package ru.yandex.solomon.util;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Sergey Polovko
 */
public enum SolomonEnv {
    DEVELOPMENT,
    PRODUCTION,
    PRESTABLE,
    TESTING,
    CLOUD_PROD,
    CLOUD_PREPROD,
    ISRAEL,
    ;

    private static final Logger logger = LoggerFactory.getLogger(SolomonEnv.class);
    private static final SolomonEnv[] VALUES = values();

    private final String id;

    SolomonEnv() {
        this.id = name().toLowerCase().replace('_', '-');
    }

    public String id() {
        return id;
    }

    public boolean isActive() {
        return this == current();
    }

    public static SolomonEnv current() {
        return EnvHolder.CURRENT;
    }

    @Nullable
    public static SolomonEnv findById(String id) {
        for (SolomonEnv e : VALUES) {
            if (e.id.equals(id)) {
                return e;
            }
        }
        return null;
    }

    /**
     * Determines environment type on first access.
     */
    private static final class EnvHolder {
        static final SolomonEnv CURRENT = resolveEnv();

        private static SolomonEnv resolveEnv() {
            try {
                return Optional.ofNullable(fromEnvironmentVariable())
                        .or(() -> Optional.ofNullable(fromFile()))
                        .orElse(DEVELOPMENT);
            } catch (Throwable e) {
                // do not silently eat exception
                // (logger may not be initialized yet)
                e.printStackTrace();
                logger.error("Failed read /etc/solomon/env", e);
                return DEVELOPMENT;
            }
        }

        @Nullable
        private static SolomonEnv fromEnvironmentVariable() {
            try {
                return Optional.ofNullable(System.getenv("SOLOMON_ENV"))
                        .map(StringUtils::stripToNull)
                        .map(SolomonEnv::findById)
                        .orElse(null);
            } catch (Throwable e) {
                // do not silently eat exception
                // (logger may not be initialized yet)
                e.printStackTrace();
                logger.error("Failed read $SOLOMON_ENV", e);
                return null;
            }
        }

        @Nullable
        private static SolomonEnv fromFile() {
            try {
                String id = StringUtils.strip(Files.readString(Path.of("/etc/solomon/env")));
                return SolomonEnv.findById(id);
            } catch (Throwable e) {
                // do not silently eat exception
                // (logger may not be initialized yet)
                e.printStackTrace();
                logger.error("Failed read /etc/solomon/env", e);
                return null;
            }
        }
    }

}
