package ru.yandex.solomon.util;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
@NotThreadSafe
public class TimeSpentInStates<E extends Enum<E>> implements MetricSupplier {
    private final String name;
    private final E[] enumValues;
    private final EnumMapToLong<E> timeSpentInStatesNanos;
    private long lastSwitchTimeNanos = 0;
    private E lastState = null;

    public TimeSpentInStates(Class<E> enumClass, String name) {
        this.timeSpentInStatesNanos = new EnumMapToLong<>(enumClass);
        this.enumValues = enumClass.getEnumConstants();
        this.name = name;
    }

    public void switchToState(E state) {
        final long nowNanos = System.nanoTime();
        if (lastSwitchTimeNanos != 0) {
            final long dtNanos = nowNanos - lastSwitchTimeNanos;
            timeSpentInStatesNanos.addAndGet(lastState, dtNanos);
        }
        lastSwitchTimeNanos = nowNanos;
        lastState = state;
    }

    @Override
    public int estimateCount() {
        return enumValues.length;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        for (E e : enumValues) {
            consumer.onMetricBegin(MetricType.RATE);
            {
                consumer.onLabelsBegin(commonLabels.size() + 2);
                commonLabels.forEach(consumer::onLabel);
                consumer.onLabel("sensor", name);
                consumer.onLabel("state", e.name());
                consumer.onLabelsEnd();
            }
            consumer.onLong(0, TimeUnit.NANOSECONDS.toMillis(timeSpentInStatesNanos.get(e)));
            consumer.onMetricEnd();
        }
    }

    public void combine(TimeSpentInStates<E> target) {
        for (E e : enumValues) {
            timeSpentInStatesNanos.addAndGet(e, target.timeSpentInStatesNanos.get(e));
        }
    }
}
