package ru.yandex.solomon.util.actors;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.misc.actor.ActorRunnerImpl;
import ru.yandex.misc.actor.Tasks;
import ru.yandex.misc.concurrent.CompletableFutures;


/**
 * Actor runner that allows to run async operations and limits the number of them.
 *
 * @author Sergey Polovko
 */
public class AsyncActorRunner {

    private final Tasks tasks = new Tasks();
    private final AsyncActorBody body;
    private final Executor executor;

    private final AtomicInteger inFlight = new AtomicInteger();
    private final int maxInFlight;

    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
    private volatile boolean done = false;


    public AsyncActorRunner(AsyncActorBody body, Executor executor, int maxInFlight) {
        this.body = body;
        this.executor = executor;
        this.maxInFlight = maxInFlight;
    }

    public CompletableFuture<Void> start() {
        if (tasks.addTask()) {
            ActorRunnerImpl.schedule(executor, this::mainLoop);
        }
        return doneFuture;
    }

    public void stop() {
        doneFuture.cancel(false);
    }

    private void mainLoop() {
        while (!doneFuture.isDone() && tasks.fetchTask()) {
            // run async operation of the actor
            boolean runOneMore = inFlight.incrementAndGet() < maxInFlight;

            CompletableFuture<?> operationFuture;
            try {
                operationFuture = body.run();
            } catch (Throwable t) {
                operationFuture = CompletableFuture.failedFuture(t);
            }

            // fast pass for already completed operation
            if (CompletableFutures.isCompletedSuccessfully(operationFuture)) {
                if (operationFuture.getNow(null) == AsyncActorBody.DONE_MARKER) {
                    done = true;
                }
                boolean last = inFlight.decrementAndGet() == 0;
                if (done) {
                    if (last) {
                        doneFuture.complete(null);
                    }
                    break;
                } else {
                    // got task already completed, get one more, and wait done marker
                    tasks.addTask();
                    continue;
                }
            }

            // there are three possible outcomes of fired async operation
            //    1) failed with an error
            //    2) successfully completed with DONE_MARKER
            //    3) successfully completed with some result

            operationFuture.whenComplete((r, t) -> {
                // remember seen done marker
                if (r == AsyncActorBody.DONE_MARKER) {
                    done = true;
                }
                boolean last = inFlight.decrementAndGet() == 0;

                // on error stop the actor as fast as possible
                if (t != null) {
                    done = true;
                    doneFuture.completeExceptionally(t);
                    return;
                }

                // if this or any neighboring async operation is done
                if (done) {
                    if (last) {
                        doneFuture.complete(null);
                    }
                } else if (tasks.addTask()) {
                    // schedule one more async operation
                    ActorRunnerImpl.schedule(executor, this::mainLoop);
                }
            });

            if (runOneMore) {
                tasks.addTask();
            }
        }
    }
}
