package ru.yandex.solomon.util.actors;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.IntFunction;

import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.time.DurationUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public class PingActorRunner implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(PingActorRunner.class);

    private final IntFunction<CompletableFuture<?>> fn;
    private final ScheduledExecutorService timer;
    private final long backoffDelayMillis;
    private final long backoffMaxDelayMillis;
    private final long pingIntervalMillis;
    private final String operation;
    private final ActorWithFutureRunner actor;

    private volatile boolean closed;
    private int attempt;
    private volatile ScheduledFuture<?> scheduled;
    private final AtomicReference<CompletableFuture<Object>> nextPing = new AtomicReference<>(new CompletableFuture<>());

    // for manager ui
    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;
    Instant lastSuccessTime = Instant.EPOCH;

    private PingActorRunner(Builder builder) {
        this.fn = builder.fn;
        this.timer = builder.timer;
        this.backoffDelayMillis = builder.backoffDelayMillis;
        this.backoffMaxDelayMillis = builder.backoffMaxDelayMillis != 0
                ? builder.backoffMaxDelayMillis
                : builder.pingIntervalMillis;
        this.pingIntervalMillis = builder.pingIntervalMillis;
        this.operation = builder.operation;
        this.actor = new ActorWithFutureRunner(this::act, builder.executor);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public void schedule() {
        var scheduled = schedule(DurationUtils.randomize(pingIntervalMillis));
        if (this.scheduled == null) {
            this.scheduled = scheduled;
        }
    }

    public CompletableFuture<Void> forcePing() {
        var future = nextPing.get();
        this.actor.schedule();
        return future.thenAccept(o -> {});
    }

    private CompletableFuture<?> act() {
        if (scheduled != null) {
            scheduled.cancel(false);
        }

        if (closed) {
            nextPing.get().completeExceptionally(new CancellationException());
            return CompletableFuture.completedFuture(null);
        }

        var doneFuture = nextPing.getAndSet(new CompletableFuture<>());
        var future = CompletableFutures.safeCall(() -> fn.apply(attempt))
                .handle((ignore, e) -> {
                    if (closed) {
                        nextPing.get().completeExceptionally(new CancellationException());
                        return null;
                    }

                    if (e != null) {
                        lastError = e;
                        lastErrorTime = Instant.now();
                        long delayMillis = backoffTimeMillis(attempt++);
                        logger.warn("{} failed, retry after {} ms", operation, delayMillis, e);
                        scheduled = timer.schedule(actor::schedule, delayMillis, TimeUnit.MILLISECONDS);
                    } else {
                        attempt = 0;
                        lastSuccessTime = Instant.now();
                        scheduled = schedule(DurationUtils.randomize(pingIntervalMillis));
                    }
                    return null;
                });
        CompletableFutures.whenComplete(future, doneFuture);
        return future;
    }

    private ScheduledFuture<?> schedule(long delayMillis) {
        return timer.schedule(actor::schedule, delayMillis, TimeUnit.MILLISECONDS);
    }

    private long backoffTimeMillis(int attempt) {
        long backoff = DurationUtils.backoff(backoffDelayMillis, backoffMaxDelayMillis, attempt);
        return DurationUtils.randomize(backoff);
    }

    @Override
    public void close() {
        closed = true;
        // race at this place ok
        var copy = scheduled;
        if (copy != null) {
            copy.cancel(false);
        }

        nextPing.get().completeExceptionally(new CancellationException());
    }

    public static class Builder {
        private IntFunction<CompletableFuture<?>> fn;
        private long backoffDelayMillis = 100;
        private long backoffMaxDelayMillis = 0;
        private long pingIntervalMillis = TimeUnit.SECONDS.toMillis(15);
        private Executor executor = MoreExecutors.directExecutor();
        private ScheduledExecutorService timer;
        private String operation = "ping";

        public Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        public Builder backoffDelay(Duration duration) {
            checkArgument(!duration.isNegative(), "delay(%s) is negative", duration);
            this.backoffDelayMillis = duration.toMillis();
            return this;
        }

        public Builder backoffMaxDelay(Duration duration) {
            checkArgument(!duration.isNegative(), "backoffMaxDelay(%s) is negative", duration);
            this.backoffMaxDelayMillis = duration.toMillis();
            return this;
        }

        public Builder pingInterval(Duration duration) {
            checkArgument(!duration.isNegative(), "pingInterval(%s) is negative", duration);
            this.pingIntervalMillis = duration.toMillis();
            return this;
        }

        public Builder executor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder timer(ScheduledExecutorService timer) {
            this.timer = timer;
            return this;
        }

        public Builder onPing(IntFunction<CompletableFuture<?>> fn) {
            this.fn = fn;
            return this;
        }

        public PingActorRunner build() {
            checkNotNull(fn, "fn for ping not defined");
            checkNotNull(timer, "timer not defined");
            return new PingActorRunner(this);
        }
    }
}
