package ru.yandex.solomon.util.actors;

import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.actor.ActorRunner;

/**
 * Runs {@link Runnable} tasks using {@link ActorRunner} and processes tasks in submitted order.
 * Task number executed in one tick limited by tickLimit field.
 *
 * Howto usage scenario based on example https://github.com/stepancheg/netty-td/blob/master/src/com/github/stepancheg/nomutex/tasks/framework/SimpleSingleQueueActor.java
 *
 * Task number executed in one tick limited by limit field.
 * It can be used when your tasks must be in synchronization, but you do not want to do it.
 * The alternative is {@link io.grpc.SynchronizationContext}.
 * Differences between this implementation and {@link io.grpc.SynchronizationContext}:
 * 1. calculates size with LongAdder for monitoring
 * 2. does not provide scheduling with delay function
 * 3. has task limit for one execution
 * 4. it does not have drain/flush/close, because it is expected that you don't care. It means that if you close your application, you do not care will be these tasks executed.
 *
 * @author Egor Litvinenko
 * */
@ParametersAreNonnullByDefault
public class SimpleQueueActor<TMessage> {

    private final Queue<TMessage> tasks;
    private final Consumer<TMessage> action;
    private final LongAdder size;
    private final ActorRunner actorRunner;
    private final int tickLimit;

    public SimpleQueueActor(
            Executor executor,
            Consumer<TMessage> action,
            Consumer<Throwable> exceptionHandler,
            int tickLimit)
    {
        this.action = Objects.requireNonNull(action);
        this.tickLimit = tickLimit;
        this.size = new LongAdder();
        this.tasks = new ConcurrentLinkedQueue<>();
        this.actorRunner = new ActorRunner(this::act, executor, exceptionHandler, 1);
    }

    public void enqueue(TMessage message) {
        tasks.add(message);
        size.increment();
        actorRunner.schedule();
    }

    public int size() {
        return size.intValue();
    }

    public void act() {
        TMessage message = tasks.poll();
        if (message == null) {
            return;
        }
        int i = 0;
        do {
            size.decrement();
            action.accept(message);
        } while (++i < tickLimit && (message = tasks.poll()) != null);
        if (message != null) {
            actorRunner.schedule();
        }
    }
}
