package ru.yandex.solomon.util.async;

import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

/**
 * @author Sergey Polovko
 */
public final class InFlightLimiter {

    private final int max;
    private final Queue<Item> waitingQueue = new ConcurrentLinkedQueue<>();
    private final AtomicInteger current = new AtomicInteger(0);
    private final AtomicLong idGen = new AtomicLong();

    public InFlightLimiter(int max) {
        this.max = max;
    }

    public int getCurrent() {
        return current.get();
    }

    public int getWaitingCount() {
        return waitingQueue.size();
    }

    public long run(Supplier<CompletableFuture<?>> fn) {
        long id = idGen.incrementAndGet();
        if (acquire()) {
            runAcquired(fn);
        } else {
            waitingQueue.offer(new Item(id, fn));
        }
        return id;
    }

    public void remove(long id) {
        waitingQueue.removeIf(item -> item.id == id);
    }

    private void runAcquired(Supplier<CompletableFuture<?>> fn) {
        try {
            fn.get().whenComplete((aVoid, throwable) -> {
                release();
                runWaiting();
            });
        } catch (Throwable t) {
            release();
            runWaiting();
        }
    }

    private void runWaiting() {
        while (true) {
            if (!acquire()) {
                break;
            }

            Item item = waitingQueue.poll();
            if (item != null) {
                runAcquired(item.fn);
            } else {
                release();
                break;
            }
        }
    }

    private boolean acquire() {
        int current;
        do {
            current = this.current.get();
            if (current >= max) {
                return false;
            }
        } while (!this.current.compareAndSet(current, current + 1));

        return true;
    }

    private void release() {
        current.decrementAndGet();
    }

    private static final class Item {
        final long id;
        final Supplier<CompletableFuture<?>> fn;

        Item(long id, Supplier<CompletableFuture<?>> fn) {
            this.id = id;
            this.fn = fn;
        }
    }
}
