package ru.yandex.solomon.util.client;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.concurrent.CompletableFutures;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class ClientFutures {

    private ClientFutures() {
    }

    /**
     * Waits for the completion of all given futures in the best effort manner.
     * <p>
     * Returns a list of results if all futures are completed normally.
     * After the first completion with OK status it awaits for other futures for {@code extraAwaitMillis}.
     * After the timeout, it will proceed with results of the futures
     * that have been already completed normally to this time, ignoring an outcome of the rest futures.
     * <p>
     * In the presence of exceptionally completed futures, the resulting future will be completed exceptionally
     * <b>iff all futures have been done and no future has been completed with OK status.</b>
     * Otherwise, after the first completion with OK status it will return the results of the futures,
     * that have been already completed normally to this time, ignoring an outcome of the rest futures
     * (including exceptionally completed ones).
     * <p>
     * If the future result implements {@link StatusAware} then OK status is determined by {@link StatusAware#isOk()}.
     * Otherwise, any normally completed future is considered as completed with OK status.
     */
    public static <T> CompletableFuture<List<T>> allOrAnyOkOf(
        List<CompletableFuture<T>> futures,
        LongSupplier extraAwaitMillis)
    {
        if (futures.isEmpty()) {
            return completedFuture(List.of());
        }

        if (futures.size() == 1) {
            return futures.get(0).thenApply(List::of);
        }

        var allOfFuture = CompletableFutures.allOf(futures);

        return anyOk(futures).thenCompose(someIsOk -> {
            if (someIsOk) {
                return allOfFuture
                    .exceptionally(t -> null)
                    // wait some more time for lagging futures
                    .completeOnTimeout(null, extraAwaitMillis.getAsLong(), TimeUnit.MILLISECONDS)
                    .thenApply(list -> {
                        if (list != null) {
                            return list;
                        }
                        var completed = new ArrayList<T>(futures.size());
                        for (var future : futures) {
                            if (future.isDone() && !future.isCompletedExceptionally()) {
                                completed.add(future.getNow(null));
                            }
                        }
                        return completed;
                    });
            }

            // All futures are completed by the time
            return allOfFuture.thenApply(listF -> listF);
        });
    }

    /**
     * Returns true when any of the futures completes to {@link StatusAware#isOk(Object)} == true
     * <p>
     * Returns false when all the futures failed or completed with {@link StatusAware#isOk(Object)} == false
     */
    private static <T> CompletableFuture<Boolean> anyOk(List<CompletableFuture<T>> futures) {
        if (futures.isEmpty()) {
            return completedFuture(false);
        }

        @SuppressWarnings("unchecked")
        CompletableFuture<T>[] futureArray = futures.toArray(CompletableFuture[]::new);

        return CompletableFuture.anyOf(futureArray)
            .exceptionally(t -> null)
            .thenCompose(ignore -> {
                // precompute to avoid race
                var isDone = futures.stream()
                    .map(CompletableFuture::isDone)
                    .toArray(Boolean[]::new);
                var incomplete = new ArrayList<CompletableFuture<T>>(futureArray.length);
                for (int i = 0; i < futureArray.length; i++) {
                    if (isDone[i]) {
                        var future = futureArray[i];
                        if (!future.isCompletedExceptionally() && StatusAware.isOk(future.getNow(null))) {
                            return completedFuture(true);
                        }
                    } else {
                        incomplete.add(futureArray[i]);
                    }
                }
                return anyOk(incomplete);
            });
    }

}
