package ru.yandex.solomon.util.collection.array;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.type.array.ObjectArrayType;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;

/**
 * @see ObjectArrayBuilder
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ArrayBuilder<A> {

    @Nonnull
    public A[] array;
    public int size;

    public ArrayBuilder(A[] array) {
        this.array = array;
        this.size = 0;
    }

    public ArrayBuilder(ObjectArrayType<A> arrayType) {
        this(arrayType.emptyArray());
    }

    public ArrayBuilder(Class<A> elementClass) {
        this((A[]) Array.newInstance(elementClass, 0));
    }

    public boolean isEmpty() {
        return size == 0;
    }

    public int remainingCapacity() {
        return array.length - size;
    }

    public void reserveAdditional(int additional) {
        if (additional > remainingCapacity()) {
            array = ArrayListImpl.reserveAdditionalInObjectArray(array, size, additional);
        }
    }

    public void add(A a) {
        reserveAdditional(1);
        addNoCheck(a);
    }

    public void addNoCheck(A a) {
        array[size] = a;
        ++size;
    }

    public void addAll(A[] add) {
        reserveAdditional(add.length);
        System.arraycopy(add, 0, array, size, add.length);
        size += add.length;
    }

    public void addAll(List<A> add) {
        addAll((A[]) add.toArray());
    }

    public void addAll(ArrayBuilder<A> add) {
        reserveAdditional(add.size);
        System.arraycopy(add.array, 0, array, size, add.size);
        size += add.size;
    }

    public boolean contains(A a) {
        for (int i = 0; i < size; ++i) {
            if (Objects.equals(this.array[i], a)) {
                return true;
            }
        }
        return false;
    }

    public void trimToSize() {
        if (array.length != size) {
            array = Arrays.copyOf(array, size);
        }
    }

    @Nonnull
    public A[] buildUnsafe() {
        if (array.length == size) {
            return array;
        }
        return Arrays.copyOf(array, size);
    }

    @Nonnull
    public A[] build() {
        A[] r = buildUnsafe();

        // make sure it is not shared
        if (array.length != 0) {
            this.array = Arrays.copyOf(array, 0);
        }

        return r;
    }

    public List<A> toList() {
        return Arrays.asList(Arrays.copyOf(array, size));
    }

}
