package ru.yandex.solomon.util.collection.array;

import java.util.stream.BaseStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class ArrayViewBase<A, E, S extends ArrayViewBase<A, E, S>> {
    public final int from;
    public final int to;

    protected ArrayViewBase(int from, int to) {
        Validate.isTrue(0 <= from);
        Validate.isTrue(from <= to);

        this.from = from;
        this.to = to;
    }

    /** Raw underlying array */
    protected abstract A array();

    protected abstract ArrayViewType<A, E, S> viewType();

    protected abstract PrimitiveArrayType<A, E> arrayType();

    public abstract BaseStream<?, ?> stream();

    public int length() {
        return to - from;
    }

    public boolean isEmpty() {
        return length() == 0;
    }

    @Nonnull
    public S take(int count) {
        if (count == 0) {
            return viewType().empty();
        } else if (count >= length()) {
            return (S) this;
        } else {
            return slice(0, count);
        }
    }

    @Nonnull
    public S drop(int count) {
        if (count == 0) {
            return (S) this;
        } else if (count >= length()) {
            return viewType().empty();
        } else {
            return slice(count, length());
        }
    }

    @Nonnull
    public S slice(int from, int to) {
        if (from == 0 && to == length()) {
            return (S) this;
        }

        if (from < 0 || from > to || to > length()) {
            throw new IllegalArgumentException("cannot slice from " + from + " to " + to);
        }

        return viewType().construct(array(), this.from + from, this.from + to);
    }

    @Override
    public int hashCode() {
        return arrayType().hashCodeOfRange(array(), from, to);
    }

    @Override
    public String toString() {
        return arrayType().toStringOfRange(array(), from, to);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null || obj.getClass() != this.getClass()) {
            return false;
        }

        ArrayViewBase<A, E, S> that = (ArrayViewBase<A, E, S>) obj;

        return arrayType().equalRanges(array(), from, to, that.array(), that.from, that.to);
    }

    public A toArray() {
        return arrayType().copyOfRange(array(), from, to);
    }

    public void copyTo(A array, int offset) {
        System.arraycopy(this.array(), this.from, array, offset, this.length());
    }
}
