package ru.yandex.solomon.util.collection.array;

import java.util.Arrays;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.booleans.BooleanArrayList;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class BooleanArrayView extends ArrayViewBase<boolean[], Boolean, BooleanArrayView> {

    public static final BooleanArrayView empty = new BooleanArrayView(Cf.BooleanArray.emptyArray(), 0, 0);

    public final boolean[] array;

    public BooleanArrayView(boolean[] array, int from, int to) {
        super(from, to);

        Validate.isTrue(to <= array.length);

        this.array = array;
    }

    public BooleanArrayView(boolean[] array) {
        this(array, 0, array.length);
    }

    public BooleanArrayView(BooleanArrayList list) {
        this(list.elements(), 0, list.size());
    }

    @Override
    protected boolean[] array() {
        return array;
    }

    public enum Type implements ArrayViewType<boolean[], Boolean, BooleanArrayView> {
        I;

        @Override
        public Class<BooleanArrayView> selfClass() {
            return BooleanArrayView.class;
        }

        @Nonnull
        @Override
        public BooleanArrayView doConstruct(boolean[] array, int from, int to) {
            return new BooleanArrayView(array, from, to);
        }

        @Nonnull
        @Override
        public BooleanArrayView empty() {
            return empty;
        }

        @Nonnull
        @Override
        public PrimitiveArrayType<boolean[], Boolean> arrayType() {
            return Cf.BooleanArray;
        }
    }

    @Override
    protected ArrayViewType<boolean[], Boolean, BooleanArrayView> viewType() {
        return Type.I;
    }


    @Override
    protected PrimitiveArrayType<boolean[], Boolean> arrayType() {
        return Cf.BooleanArray;
    }

    @Override
    public IntStream stream() {
        return IntStream.range(from, to)
            .map(i -> array[i] ? 1 : 0);
    }


    public boolean[] copyToArray() {
        if (length() == 0) {
            return Cf.BooleanArray.emptyArray();
        } else {
            return Arrays.copyOfRange(array, from, to);
        }
    }

    public boolean[] copyOrArray() {
        if (from == 0 && to == array.length) {
            return array;
        } else {
            return copyToArray();
        }
    }

    public boolean contains(boolean b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) == b) {
                return true;
            }
        }
        return false;
    }

    public boolean allAre(boolean b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) != b) {
                return false;
            }
        }
        return true;
    }

    public boolean at(int i) {
        return array[from + i];
    }

    public boolean first() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get first value in empty array");
        }

        return at(0);
    }

    public boolean last() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last value in empty array");
        }

        return at(length() - 1);
    }

}
