package ru.yandex.solomon.util.collection.array;

import java.util.Arrays;
import java.util.stream.DoubleStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.doubles.DoubleArrayList;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class DoubleArrayView extends ArrayViewBase<double[], Double, DoubleArrayView> {

    public static final DoubleArrayView empty = new DoubleArrayView(Cf.DoubleArray.emptyArray(), 0, 0);

    public final double[] array;

    public DoubleArrayView(double[] array, int from, int to) {
        super(from, to);

        Validate.isTrue(to <= array.length);

        this.array = array;
    }

    public DoubleArrayView(double[] array) {
        this(array, 0, array.length);
    }

    public DoubleArrayView(DoubleArrayList list) {
        this(list.elements(), 0, list.size());
    }

    @Override
    protected double[] array() {
        return array;
    }

    public enum Type implements ArrayViewType<double[], Double, DoubleArrayView> {
        I;

        @Override
        public Class<DoubleArrayView> selfClass() {
            return DoubleArrayView.class;
        }

        @Nonnull
        @Override
        public DoubleArrayView doConstruct(double[] array, int from, int to) {
            return new DoubleArrayView(array, from, to);
        }

        @Nonnull
        @Override
        public DoubleArrayView empty() {
            return empty;
        }

        @Nonnull
        @Override
        public PrimitiveArrayType<double[], Double> arrayType() {
            return Cf.DoubleArray;
        }
    }

    @Override
    protected Type viewType() {
        return Type.I;
    }


    @Override
    protected PrimitiveArrayType<double[], Double> arrayType() {
        return Cf.DoubleArray;
    }


    @Override
    public DoubleStream stream() {
        return Arrays.stream(array, from, to);
    }

    public double[] copyToArray() {
        if (length() == 0) {
            return Cf.DoubleArray.emptyArray();
        } else {
            return Arrays.copyOfRange(array, from, to);
        }
    }

    public double[] copyOrArray() {
        if (from == 0 && to == array.length) {
            return array;
        } else {
            return copyToArray();
        }
    }

    public double at(int i) {
        return array[from + i];
    }

    public boolean contains(double b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) == b) {
                return true;
            }
        }
        return false;
    }

    public boolean allAre(double b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) != b) {
                return false;
            }
        }
        return true;
    }

    public double first() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get first value in empty array");
        }

        return at(0);
    }

    public double last() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last value in empty array");
        }

        return at(length() - 1);
    }

}
