package ru.yandex.solomon.util.collection.array;

import java.util.Arrays;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.IntArrayList;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class IntArrayView extends ArrayViewBase<int[], Integer, IntArrayView> {

    public static final IntArrayView empty = new IntArrayView(Cf.IntegerArray.emptyArray(), 0, 0);

    public final int[] array;

    public IntArrayView(int[] array, int from, int to) {
        super(from, to);

        Validate.isTrue(to <= array.length);

        this.array = array;
    }

    public IntArrayView(int[] array) {
        this(array, 0, array.length);
    }

    public IntArrayView(IntArrayList list) {
        this(list.elements(), 0, list.size());
    }

    @Override
    protected int[] array() {
        return array;
    }

    public enum Type implements ArrayViewType<int[], Integer, IntArrayView> {
        I;

        @Override
        public Class<IntArrayView> selfClass() {
            return IntArrayView.class;
        }

        @Nonnull
        @Override
        public IntArrayView doConstruct(int[] array, int from, int to) {
            return new IntArrayView(array, from, to);
        }

        @Nonnull
        @Override
        public IntArrayView empty() {
            return empty;
        }

        @Nonnull
        @Override
        public PrimitiveArrayType<int[], Integer> arrayType() {
            return Cf.IntegerArray;
        }
    }

    @Override
    protected Type viewType() {
        return Type.I;
    }


    @Override
    protected PrimitiveArrayType<int[], Integer> arrayType() {
        return Cf.IntegerArray;
    }


    @Override
    public IntStream stream() {
        return Arrays.stream(array, from, to);
    }

    @Override
    public void copyTo(int[] array, int offset) {
        System.arraycopy(this.array(), this.from, array, offset, this.length());
    }

    public int[] copyToArray() {
        if (length() == 0) {
            return Cf.IntegerArray.emptyArray();
        } else {
            return Arrays.copyOfRange(array, from, to);
        }
    }

    public int[] copyOrArray() {
        if (from == 0 && to == array.length) {
            return array;
        } else {
            return copyToArray();
        }
    }

    public int at(int i) {
        return array[from + i];
    }

    public boolean contains(int b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) == b) {
                return true;
            }
        }
        return false;
    }

    public boolean allAre(int b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) != b) {
                return false;
            }
        }
        return true;
    }

    public int first() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get first value in empty array");
        }

        return at(0);
    }

    public int last() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last value in empty array");
        }

        return at(length() - 1);
    }

}
