package ru.yandex.solomon.util.collection.array;

import java.util.Arrays;
import java.util.stream.LongStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.LongArrayList;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LongArrayView extends ArrayViewBase<long[], Long, LongArrayView> {

    public static final LongArrayView empty = new LongArrayView(Cf.LongArray.emptyArray(), 0, 0);

    public final long[] array;

    public LongArrayView(long[] array, int from, int to) {
        super(from, to);

        Validate.isTrue(to <= array.length);

        this.array = array;
    }

    public LongArrayView(long[] array) {
        this(array, 0, array.length);
    }

    public LongArrayView(LongArrayList list) {
        this(list.elements(), 0, list.size());
    }

    @Override
    protected long[] array() {
        return array;
    }

    public enum Type implements ArrayViewType<long[], Long, LongArrayView> {
        I;

        @Override
        public Class<LongArrayView> selfClass() {
            return LongArrayView.class;
        }

        @Nonnull
        @Override
        public LongArrayView doConstruct(long[] array, int from, int to) {
            return new LongArrayView(array, from, to);
        }

        @Nonnull
        @Override
        public LongArrayView empty() {
            return empty;
        }

        @Nonnull
        @Override
        public PrimitiveArrayType<long[], Long> arrayType() {
            return Cf.LongArray;
        }
    }

    @Override
    protected Type viewType() {
        return Type.I;
    }

    @Override
    protected PrimitiveArrayType<long[], Long> arrayType() {
        return Cf.LongArray;
    }

    @Override
    public LongStream stream() {
        return Arrays.stream(array, from, to);
    }

    public long[] copyToArray() {
        if (length() == 0) {
            return Cf.LongArray.emptyArray();
        } else {
            return Arrays.copyOfRange(array, from, to);
        }
    }

    public long[] copyOrArray() {
        if (from == 0 && to == array.length) {
            return array;
        } else {
            return copyToArray();
        }
    }

    public long at(int i) {
        return array[from + i];
    }

    public boolean contains(long b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) == b) {
                return true;
            }
        }
        return false;
    }

    public boolean allAre(long b) {
        for (int i = 0; i < length(); ++i) {
            if (at(i) != b) {
                return false;
            }
        }
        return true;
    }

    public long first() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get first value in empty array");
        }

        return at(0);
    }

    public long last() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last value in empty array");
        }

        return at(length() - 1);
    }

}
