package ru.yandex.solomon.util.collection.enums;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.function.BiFunction;
import java.util.function.ToLongBiFunction;

import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.solomon.util.collection.LikeMap;
import ru.yandex.solomon.util.reflect.reified.ReifiedToString;

/**
 * @author Stepan Koltsov
 */
abstract class EnumMapTo<E extends Enum<E>, A> implements ReifiedToString<E> {

    @Override
    public String toString() {
        StringBuilder r = new StringBuilder();
        r.append("{");
        for (int i = 0; i < size(); ++i) {
            if (i != 0) {
                r.append(", ");
            }
            // enum names are not known at this point
            appendValueForKey(i, r);
        }
        r.append("}");
        return r.toString();
    }

    @Override
    public String toString(Class<E> enumClass) {
        return Arrays.stream(enumClass.getEnumConstants())
            .map(e -> new AbstractMap.SimpleEntry<>(e, valueForKeyToString(e.ordinal())))
            .collect(LikeMap.collectorToString());
    }

    protected abstract int size();

    protected abstract A getValueForKey(int keyOrdinal);

    protected abstract void appendValueForKey(int keyOrdinal, StringBuilder sb);

    protected String valueForKeyToString(int keyOrdinal) {
        StringBuilder r = new StringBuilder();
        appendValueForKey(keyOrdinal, r);
        return r.toString();
    }

    public <V> EnumMap<E, V> map(Class<E> enumClass, BiFunction<E, A, V> f) {
        return EnumMapUtils.fill(enumClass, e -> {
            return f.apply(e, getValueForKey(e.ordinal()));
        });
    }

    public EnumMapToLong<E> mapToLong(Class<E> enumClass, ToLongBiFunction<E, A> f) {
        EnumMapToLong<E> result = new EnumMapToLong<>(enumClass);
        E[] enumConstants = enumClass.getEnumConstants();
        for (int i = 0; i < enumConstants.length; i++) {
            E e = enumConstants[i];
            result.set(e, f.applyAsLong(e, getValueForKey(i)));
        }
        return result;
    }
}
