package ru.yandex.solomon.util.collection.enums;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class EnumMapToAnyInt<E extends Enum<E>> extends EnumMapTo<E, Integer> {
    public abstract void set(E e, int value);

    public int get(E e) {
        return getForOrdinal(e.ordinal());
    }

    public abstract int getForOrdinal(int ordinal);

    @Override
    protected Integer getValueForKey(int keyOrdinal) {
        return getForOrdinal(keyOrdinal);
    }

    public int incrementAndGet(E e) {
        return addAndGet(e, 1);
    }

    public abstract int addAndGet(E e, int delta);

    public abstract int getAndSet(E e, int value);

    public int sumValues() {
        int sum = 0;
        for (int i = 0; i < size(); ++i) {
            sum += getForOrdinal(i);
        }
        return sum;
    }

    public static class Entry<E extends Enum<E>> {
        private final E e;
        private final int value;

        public Entry(E e, int value) {
            this.e = e;
            this.value = value;
        }

        public E getE() {
            return e;
        }

        public int getValue() {
            return value;
        }
    }

    public Stream<Entry<E>> entryStream(Class<E> eClass) {
        return Arrays.stream(eClass.getEnumConstants())
            .map(e -> new Entry<>(e, get(e)));
    }

    @Override
    public final boolean equals(Object obj) {
        if (!(obj instanceof EnumMapToAnyInt<?>)) {
            return false;
        }
        EnumMapToAnyInt<?> that = (EnumMapToAnyInt<?>) obj;
        if (this.size() != that.size()) {
            return false;
        }
        for (int i = 0; i < size(); ++i) {
            if (this.getForOrdinal(i) != that.getForOrdinal(i)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 1;
        for (int i = 0; i < this.size(); ++i) {
            hash = 31 * hash + Integer.hashCode(this.getForOrdinal(i));
        }
        return hash;
    }

    public Map<E, Integer> toMap(Class<E> clazz) {
        return entryStream(clazz).collect(Collectors.toMap(Entry::getE, Entry::getValue));
    }

    public Map<E, Integer> toMapNoZeroes(Class<E> clazz) {
        return entryStream(clazz).filter(entry -> entry.getValue() != 0).collect(Collectors.toMap(Entry::getE, Entry::getValue));
    }
}
