package ru.yandex.solomon.util.collection.enums;

import java.util.concurrent.atomic.AtomicLongArray;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class EnumMapToAtomicLong<E extends Enum<E>> extends EnumMapToAnyLong<E> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(EnumMapToInt.class);
    private final AtomicLongArray values;

    public EnumMapToAtomicLong(Class<E> enumClass) {
        if (!enumClass.isEnum()) {
            throw new IllegalArgumentException("class is not enum: " + enumClass.getSimpleName());
        }
        values = new AtomicLongArray(enumClass.getEnumConstants().length);
    }

    @Override
    public void set(E e, long value) {
        values.set(e.ordinal(), value);
    }

    @Override
    public long getForOrdinal(int ordinal) {
        return values.get(ordinal);
    }

    @Override
    public long addAndGetForOrdinal(int ordinal, long delta) {
        return values.addAndGet(ordinal, delta);
    }

    public long getAndSet(E e, long l) {
        return values.getAndSet(e.ordinal(), l);
    }

    @Override
    protected int size() {
        return values.length();
    }

    @Override
    protected void appendValueForKey(int keyOrdinal, StringBuilder sb) {
        sb.append(values.get(keyOrdinal));
    }

    public EnumMapToLong<E> snapshot() {
        return new EnumMapToLong<>(this);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + (MemoryCounter.ARRAY_HEADER_SIZE + values.length() * Long.BYTES);
    }
}
