package ru.yandex.solomon.util.collection.enums;

import java.util.Arrays;
import java.util.stream.Collector;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class EnumMapToInt<E extends Enum<E>> extends EnumMapToAnyInt<E> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(EnumMapToInt.class);
    private final int[] values;

    public EnumMapToInt(Class<E> enumClass) {
        if (!enumClass.isEnum()) {
            throw new IllegalArgumentException("class is not enum: " + enumClass.getSimpleName());
        }
        values = new int[enumClass.getEnumConstants().length];
    }

    public EnumMapToInt(EnumMapToInt<E> map) {
        this.values = new int[map.size()];
        for (int i = 0; i < values.length; ++i) {
            values[i] = map.getForOrdinal(i);
        }
    }

    public EnumMapToInt(Class<E> enumClass, int defaultValue) {
        this(enumClass);
        Arrays.fill(values, defaultValue);
    }

    @Override
    public void set(E e, int value) {
        values[e.ordinal()] = value;
    }

    @Override
    public int getForOrdinal(int ordinal) {
        return values[ordinal];
    }

    @Override
    public int addAndGet(E e, int delta) {
        return values[e.ordinal()] += delta;
    }

    @Override
    public int getAndSet(E e, int l) {
        int r = get(e);
        set(e, l);
        return r;
    }

    public void addAll(EnumMapToInt<E> that) {
        for (int i = 0; i < values.length; ++i) {
            this.values[i] += that.values[i];
        }
    }

    @Override
    protected int size() {
        return values.length;
    }

    @Override
    protected void appendValueForKey(int keyOrdinal, StringBuilder sb) {
        sb.append(values[keyOrdinal]);
    }

    public static <E extends Enum<E>> Collector<E, ?, EnumMapToInt<E>> collector(Class<E> clazz) {
        return Collector.of(
            () -> new EnumMapToInt<E>(clazz),
            (a, e) -> a.addAndGet(e, 1),
            (a, b) -> { a.addAll(b); return a; });
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.arrayObjectSize(values);
    }
}
