package ru.yandex.solomon.util.collection.enums;

import java.util.Arrays;
import java.util.List;
import java.util.function.ToLongFunction;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class EnumMapToLong<E extends Enum<E>> extends EnumMapToAnyLong<E> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(EnumMapToLong.class);
    private final long[] values;

    public EnumMapToLong(Class<E> enumClass) {
        if (!enumClass.isEnum()) {
            throw new IllegalArgumentException("class is not enum: " + enumClass.getSimpleName());
        }
        values = new long[enumClass.getEnumConstants().length];
    }

    private EnumMapToLong(long[] values) {
        this.values = values;
    }

    public EnumMapToLong(EnumMapToAnyLong<E> map) {
        this.values = new long[map.size()];
        for (int i = 0; i < values.length; ++i) {
            values[i] = map.getForOrdinal(i);
        }
    }

    public EnumMapToLong(Class<E> enumClass, long defaultValue) {
        this(enumClass);
        Arrays.fill(values, defaultValue);
    }

    public EnumMapToLong(Class<E> enumClass, ToLongFunction<E> init) {
        this(enumClass);
        for (E e : enumClass.getEnumConstants()) {
            values[e.ordinal()] = init.applyAsLong(e);
        }
    }

    @Override
    public void set(E e, long value) {
        values[e.ordinal()] = value;
    }

    @Override
    public long getForOrdinal(int ordinal) {
        return values[ordinal];
    }

    @Override
    public long addAndGetForOrdinal(int ordinal, long delta) {
        return values[ordinal] += delta;
    }

    public long getAndSet(E e, long l) {
        long r = get(e);
        set(e, l);
        return r;
    }

    public void addAll(EnumMapToLong<E> that) {
        for (int i = 0; i < values.length; ++i) {
            this.values[i] += that.values[i];
        }
    }

    @Override
    protected int size() {
        return values.length;
    }

    @Override
    protected void appendValueForKey(int keyOrdinal, StringBuilder sb) {
        sb.append(values[keyOrdinal]);
    }

    public long sumValues() {
        return Arrays.stream(values).sum();
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.arrayObjectSize(values);
    }

    @Nonnull
    public static <E extends Enum<E>> EnumMapToLong<E> sum(List<EnumMapToLong<E>> shardAddendums, Class<E> enumClass) {
        return shardAddendums.stream().collect(sumCollector(enumClass));
    }

    public static <E extends Enum<E>> Collector<EnumMapToLong<E>, ?, EnumMapToLong<E>> sumCollector(Class<E> clazz) {
        return Collectors.reducing(new EnumMapToLong<>(clazz), (a, b) -> {
            a.addAll(b);
            return a;
        });
    }

    public static <E extends Enum<E>> Collector<E, ?, EnumMapToLong<E>> collector(Class<E> clazz) {
        return Collector.of(
            () -> new EnumMapToLong<E>(clazz),
            (a, e) -> a.addAndGet(e, 1),
            (a, b) -> { a.addAll(b); return a; });
    }

    @Nonnull
    public EnumMapToLong<E> copy() {
        return new EnumMapToLong<>(values.clone());
    }
}
