package ru.yandex.solomon.util.collection.enums;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.collection.LikeCollection;
import ru.yandex.solomon.util.reflect.reified.ReifiedToString;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class EnumSetAtomic<E extends Enum<E>> implements ReifiedToString<E> {

    private volatile long bits = 0;
    // TODO: support more than 64 values

    private static final AtomicLongFieldUpdater<EnumSetAtomic> bitsField =
        AtomicLongFieldUpdater.newUpdater(EnumSetAtomic.class, "bits");

    private EnumSetAtomic(Class<E> enumClass) {
        if (enumClass.getEnumConstants().length >= 64) {
            throw new RuntimeException("TODO");
        }
    }

    public static <E extends Enum<E>> EnumSetAtomic<E> noneOf(Class<E> enumClass) {
        return new EnumSetAtomic<>(enumClass);
    }

    public boolean has(E value) {
        return (bits & (1 << value.ordinal())) != 0;
    }

    public void set(E value) {
        for (;;) {
            long old = this.bits;
            long updated = old | (1 << value.ordinal());
            if (updated == old) {
                return;
            }
            if (bitsField.weakCompareAndSet(this, old, updated)) {
                return;
            }
        }
    }

    public void clear(E value) {
        for (;;) {
            long old = this.bits;
            long updated = old & ~(1 << value.ordinal());
            if (updated == old) {
                return;
            }
            if (bitsField.weakCompareAndSet(this, old, updated)) {
                return;
            }
        }
    }

    @Nonnull
    public EnumSet<E> toEnumSet(Class<E> enumClass) {
        EnumSet<E> r = EnumSet.noneOf(enumClass);
        for (E e : enumClass.getEnumConstants()) {
            if (has(e)) {
                r.add(e);
            }
        }
        return r;
    }

    @Override
    public String toString(Class<E> enumClass) {
        return Arrays.stream(enumClass.getEnumConstants())
            .filter(this::has)
            .collect(LikeCollection.collectorToString());
    }
}
