package ru.yandex.solomon.util.collection.queue;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.locks.ReentrantLock;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class ArrayListLockQueue<T> implements MemMeasurable {

    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ArrayListLockQueue.class);

    private static final int MIN_INITIAL_CAPACITY = 10;
    private static final int DEFAULT_INITIAL_CAPACITY = 100;

    private ArrayList<T> items;
    private long memoryUse;
    private int initialCapacity;
    private final ReentrantLock lock = new ReentrantLock();


    public ArrayListLockQueue() {
        this(DEFAULT_INITIAL_CAPACITY);
    }

    public ArrayListLockQueue(int initialCapacity) {
        this.initialCapacity = initialCapacity;
        this.items = new ArrayList<>(initialCapacity);
    }

    public void enqueue(T t) {
        lock.lock();
        try {
            items.add(t);
        } finally {
            lock.unlock();
        }
    }

    @SuppressWarnings("NonAtomicOperation")
    public void enqueueAll(Collection<T> ts) {
        long mem = 0;
        for (var value : ts) {
            mem += estimateMemory(value);
        }
        lock.lock();
        try {
            items.addAll(ts);
            memoryUse += mem;
        } finally {
            lock.unlock();
        }
    }

    public boolean enqueueIfSizeIsLess(T t, int limit) {
        long mem = estimateMemory(t);
        lock.lock();
        try {
            if (items.size() >= limit) {
                return false;
            }
            items.add(t);
            memoryUse += mem;
            return true;
        } finally {
            lock.unlock();
        }
    }

    public boolean enqueueAllIfSizeLess(Collection<T> ts, int limit) {
        long mem = 0;
        for (var v : ts) {
            mem += estimateMemory(v);
        }
        lock.lock();
        try {
            if (items.size() >= limit) {
                return false;
            }
            items.addAll(ts);
            memoryUse += mem;
            return true;
        } finally {
            lock.unlock();
        }
    }

    @Nonnull
    public ArrayList<T> dequeueAll() {
        lock.lock();
        try {
            ArrayList<T> r = items;

            //
            // auto adjustment of queue initial capacity
            //
            final int nextInitialCapacity;
            if (items.size() > initialCapacity) {
                nextInitialCapacity = initialCapacity + (initialCapacity >> 1);
            } else if (items.size() < (initialCapacity >> 1)) {
                nextInitialCapacity = Math.max(MIN_INITIAL_CAPACITY, initialCapacity >> 1);
            } else {
                nextInitialCapacity = initialCapacity;
            }

            memoryUse = 0;
            items = new ArrayList<>(nextInitialCapacity);
            initialCapacity = nextInitialCapacity;
            return r;
        } finally {
            lock.unlock();
        }
    }

    public int size() {
        lock.lock();
        try {
            return items.size();
        } finally {
            lock.unlock();
        }
    }

    public int estimateSize() {
        return items.size();
    }

    public int initialCapacity() {
        lock.lock();
        try {
            return initialCapacity;
        } finally {
            lock.unlock();
        }
    }

    public void clear() {
        lock.lock();
        try {
            items.clear();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public String toString() {
        return ArrayListLockQueue.class.getSimpleName() + "[size=" + size() + "]";
    }

    @Override
    public long memorySizeIncludingSelf() {
        // usafe access to memoryUse because for monitoring estimated value ok
        return SELF_SIZE + memoryUse;
    }

    protected long estimateMemory(T object) {
        return MemoryCounter.OBJECT_POINTER_SIZE;
    }
}
