package ru.yandex.solomon.util.collection.unpacked.arrayList;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.objectweb.asm.Type;

import ru.yandex.bolts.collection.impl.AbstractListF;
import ru.yandex.bolts.type.array.ArrayType;
import ru.yandex.commune.mh.Mhx;
import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.misc.MhObjenesis;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;
import ru.yandex.misc.algo.sort.ArraySortAccessorWithBuffer;
import ru.yandex.misc.algo.sort.TimSortImpl;
import ru.yandex.misc.reflection.FieldX;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.util.collection.unpacked.commonImpl.FieldNameAndType;
import ru.yandex.solomon.util.collection.unpacked.commonImpl.StructOfArrays;
import ru.yandex.solomon.util.collection.unpacked.commonImpl.UnpackedCommon;

/**
 * @author Stepan Koltsov
 */
public class UnpackedArrayList<A> extends AbstractListF<A> implements MemMeasurable {
    private final Factory<A> factory;

    // struct with array field for each field of A
    private final Object bigStruct;

    private int capacity;
    private int size;

    private UnpackedArrayList(Factory<A> factory) {
        this(factory, 0);
    }

    private UnpackedArrayList(Factory<A> factory, int capacity) {
        this.factory = factory;
        this.size = 0;
        this.capacity = capacity;
        try {
            if (capacity != 0) {
                bigStruct = factory.arrays.constructorAllocate.invokeExact(capacity);
            } else {
                bigStruct = factory.arrays.constructorEmpty.invokeExact();
            }
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        return (long) capacity * factory.itemSize;
    }

    private class Accessor implements ArrayListImpl.Accessor {

        @Override
        public int capacity() {
            return capacity;
        }

        @Override
        public int size() {
            return size;
        }

        @Override
        public void reserveExact(int newCapacity) {
            try {
                factory.arrays.resize.invokeExact(bigStruct, newCapacity);
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
            capacity = newCapacity;
        }
    }

    private int remainingCapacity() {
        return capacity - size;
    }

    public void reserveAdditional(int additional) {
        if (remainingCapacity() < additional) {
            ArrayListImpl.reserveAdditional(new Accessor(), additional);
        }
    }

    public void trimToSize() {
        if (capacity > size) {
            try {
                factory.arrays.resize.invokeExact(bigStruct, size);
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
            capacity = size;
        }
    }

    @Override
    public int size() {
        return size;
    }

    public int capacity() {
        return capacity;
    }

    @Override
    public A get(int index) {
        if (index >= size) {
            throw new IllegalArgumentException("cannot get element by index: " + index);
        }

        return getFromBigStruct(bigStruct, index);
    }

    private A getFromBigStruct(Object bigStruct, int index) {
        try {
            return (A) factory.allocateAndGetGeneric.invokeExact(bigStruct, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    void swapForTest(int i, int j) {
        try {
            factory.arrays.swapInBigStruct.invokeExact(bigStruct, i, j);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public <T> T[] toArray(T[] a) {
        if (a.length < size) {
            a = Arrays.copyOf(a, size);
        }
        for (int i = 0; i < size; ++i) {
            a[i] = (T) get(i);
        }
        return a;
    }

    private void setNoCheck(int index, A value) {
        if (value == null) {
            throw new NullPointerException("cannot set no check because value is null");
        }

        try {
            factory.setGeneric.invokeExact(bigStruct, index, value);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public A set(int index, A value) {
        if (index >= size) {
            throw new IllegalArgumentException("cannot set " + index + " element to array");
        }

        setNoCheck(index, value);
        return null;
    }

    @Override
    public boolean add(A value) {
        reserveAdditional(1);

        setNoCheck(size, value);

        ++size;
        return true;
    }

    @Override
    public void addAll(A... additions) {
        reserveAdditional(additions.length);
        for (int i = 0; i < additions.length; i++) {
            setNoCheck(size + i, additions[i]);
        }
        size += additions.length;
    }

    @Override
    public boolean addAll(Collection<? extends A> c) {
        reserveAdditional(c.size());
        int i = 0;
        for (A a : c) {
            setNoCheck(size + i, a);
            i++;
        }
        size += i;
        return i != 0;
    }

    @Override
    public void sort(Comparator<? super A> c) {
        if (size <= 1) {
            return;
        }

        StructOfArrays arrays = factory.arrays;

        TimSortImpl.sort(new ArraySortAccessorWithBuffer() {
            Object buffer;

            @Override
            public void ensureBufferSize(int size) {
                try {
                    buffer = arrays.constructorAllocate.invokeExact(size);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public boolean lessArrayBuffer(int pos, int posInBuffer) {
                return c.compare(getFromBigStruct(bigStruct, pos), getFromBigStruct(buffer, posInBuffer)) < 0;
            }

            @Override
            public boolean lessBufferArray(int posInBuffer, int pos) {
                return c.compare(getFromBigStruct(buffer, posInBuffer), getFromBigStruct(bigStruct, pos)) < 0;
            }

            @Override
            public void moveBufferArray(int posInBuffer, int pos) {
                try {
                    arrays.copyBetweenStructs.invokeExact(bigStruct, pos, buffer, posInBuffer);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public void moveBufferArray(int posInBuffer, int pos, int size) {
                try {
                    arrays.copyRangeBetweenStructs.invokeExact(bigStruct, pos, buffer, posInBuffer, size);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public void moveArrayBuffer(int pos, int posInBuffer, int size) {
                try {
                    arrays.copyRangeBetweenStructs.invokeExact(buffer, posInBuffer, bigStruct, pos, size);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public boolean less(int i, int j) {
                return c.compare(getFromBigStruct(bigStruct, i), getFromBigStruct(bigStruct, j)) < 0;
            }

            @Override
            public void swap(int i, int j) {
                try {
                    arrays.swapInBigStruct.invokeExact(bigStruct, i, j);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public void move(int from, int to, int size) {
                try {
                    arrays.copyRangeInStruct.invokeExact(bigStruct, to, from, size);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }

            @Override
            public void move(int from, int to) {
                try {
                    arrays.copyInStruct.invokeExact(bigStruct, to, from);
                } catch (Throwable throwable) {
                    throw new RuntimeException(throwable);
                }
            }
        }, 0, size);
    }

    @Override
    public Stream<A> stream() {
        return IntStream.range(0, size).mapToObj(this::get);
    }

    public static class Factory<A> {

        // X
        private final StructOfArrays arrays;

        // X, int, a -> ()
        private final MethodHandle setTyped;
        private final MethodHandle setGeneric;

        // X, int -> a
        private final MethodHandle allocateAndGetTyped;
        private final MethodHandle allocateAndGetGeneric;

        private final int itemSize;

        private static class FieldAccessor<A, F> {
            protected final Field field;
            private final String name;

            private StructOfArrays.ArraysField field2;

            private final ArrayType<Object, Object> arrayType;

            // X, int, a -> ()
            private MethodHandle copyFromBigArraysToElement;
            // X, int, a -> ()
            private MethodHandle copyFromElementToArrays;

            private FieldAccessor(FieldX field, int fieldIndex) {
                name = field.getName();

                Class<Object> fieldType = field.getType().getClazz();
                arrayType = (ArrayType<Object, Object>) ArrayType.forElementType(fieldType);
                this.field = field.getField();
            }

            private void finishConstruction(Class<?> arraysStruct) {
                try {
                    Field arraysStructField = arraysStruct.getDeclaredField(name);
                    arraysStructField.setAccessible(true);

                    // X, int, a -> ()
                    copyFromBigArraysToElement = field2.makeCopyFromArrayToAny(field);

                    // X, int, a -> ()
                    copyFromElementToArrays = field2.makeCopyFromAnyToArrays(field);

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }

        private final FieldAccessor[] fields;

        private MethodHandle invokeAll(Function<FieldAccessor, MethodHandle> f) {
            return Mhx.invokeAll(Arrays.stream(fields).map(f).toArray(MethodHandle[]::new));
        }

        public Factory(Class<A> elementType) {
            List<FieldX> fields = UnpackedCommon.dataFields(elementType);

            this.fields = new FieldAccessor[fields.size()];
            for (int i = 0; i < fields.size(); i++) {
                this.fields[i] = new FieldAccessor(fields.get(i), i);
            }

            this.itemSize = fields.stream().mapToInt(f -> f.getType().getSize()).sum();

            try {
                {
                    String fieldClassName = Type.getInternalName(elementType) + "$$UnpackedArrayList$$Arrays";
                    List<FieldNameAndType> fieldst = Arrays.stream(this.fields).map(f -> {
                        return new FieldNameAndType(f.field.getName(), f.arrayType.arrayClass());
                    }).collect(Collectors.toList());
                    arrays = new StructOfArrays(
                        elementType.getClassLoader(),
                        fieldClassName,
                        fieldst);
                }

                for (FieldAccessor field : this.fields) {
                    field.field2 = arrays.fieldByName(field.name);
                    field.finishConstruction(arrays.clazz);
                }

                setTyped = invokeAll(f -> f.copyFromElementToArrays);
                setGeneric = MethodHandles.explicitCastArguments(
                    setTyped,
                    MethodType.methodType(void.class, Object.class, int.class, Object.class));

                MethodHandle getTyped = invokeAll(f -> f.copyFromBigArraysToElement);

                allocateAndGetTyped = MhBuilder.oneshot2(
                    arrays.clazz, int.class,
                    (bigStruct, i) -> {
                        MhExpr instance = MhObjenesis.newInstance(elementType);
                        return MhCall.call(getTyped, bigStruct, i, instance).andReturnParam(2);
                    });
                allocateAndGetGeneric = MethodHandles.explicitCastArguments(
                    allocateAndGetTyped,
                    MethodType.methodType(Object.class, Object.class, int.class));

            } catch (Throwable e) {
                throw new RuntimeException(e);
            }
        }

        public UnpackedArrayList<A> newInstance() {
            return new UnpackedArrayList<>(this);
        }

        public UnpackedArrayList<A> newWithCapacity(int capacity) {
            return new UnpackedArrayList<>(this, capacity);
        }

        public UnpackedArrayList<A> fromList(List<A> list) {
            UnpackedArrayList<A> r = newWithCapacity(list.size());
            r.addAll(list);
            return r;
        }
    }
}
