package ru.yandex.solomon.util.concurrent;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.concurrent.atomic.AtomicLong;

import sun.misc.Unsafe;

import ru.yandex.misc.jvm.UnsafeUtils;


/**
 * This implementation based on {@link java.util.Random} and {@link java.util.concurrent.atomic.AtomicLongFieldUpdater}.
 *
 * @author Sergey Polovko
 */
public class AtomicRandomFieldUpdater<T> {

    private static final Unsafe unsafe = UnsafeUtils.UNSAFE;

    private static final long multiplier = 0x5DEECE66DL;
    private static final long addend = 0xBL;
    private static final long mask = (1L << 48) - 1;

    private final long offset;
    private final Class<T> targetClass;


    public static <U> AtomicRandomFieldUpdater<U> newUpdater(Class<U> targetClass, String fieldName) {
        return new AtomicRandomFieldUpdater<>(fieldName, targetClass);
    }

    private AtomicRandomFieldUpdater(String fieldName, Class<T> targetClass) {
        final Field field;
        final int modifiers;
        try {
            field = targetClass.getDeclaredField(fieldName);
            modifiers = field.getModifiers();
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }

        Class<?> fieldType = field.getType();
        if (fieldType != long.class) {
            String msg = String.format("field %s of %s must be long", fieldName, targetClass.getName());
            throw new IllegalArgumentException(msg);
        }

        if (!Modifier.isVolatile(modifiers)) {
            String msg = String.format("field %s of %s is not volatile", fieldName, targetClass.getName());
            throw new IllegalArgumentException(msg);
        }

        this.targetClass = targetClass;
        offset = unsafe.objectFieldOffset(field);
    }

    // ---- RANDOM ----

    private static final AtomicLong seedUniquifier = new AtomicLong(8682522807148012L);

    private static long seedUniquifier() {
        // L'Ecuyer, "Tables of Linear Congruential Generators of
        // Different Sizes and Good Lattice Structure", 1999
        for (;;) {
            long current = seedUniquifier.get();
            long next = current * 181783497276652981L;
            if (seedUniquifier.compareAndSet(current, next))
                return next;
        }
    }

    public void setDeafultSeed(T obj) {
        set(obj, seedUniquifier() ^ System.nanoTime());
    }

    public void setSeed(T obj, long seed) {
        set(obj, (seed ^ multiplier) & mask);
    }

    public int nextInt(T obj) {
        return next(obj, 32);
    }

    public int nextInt(T obj, int bound) {
        if (bound <= 0) {
            throw new IllegalArgumentException("bound must be positive");
        }

        int r = next(obj, 31);
        int m = bound - 1;
        if ((bound & m) == 0) {  // i.e., bound is a power of 2
            r = (int) ((bound * (long) r) >> 31);
        } else {
            int u = r;
            while (u - (r = u % bound) + m < 0) {
                u = next(obj, 31);
            }
        }
        return r;
    }

    public long nextLong(T obj) {
        // it's okay that the bottom word remains signed.
        return ((long) (next(obj, 32)) << 32) + next(obj, 32);
    }

    public boolean nextBoolean(T obj) {
        return next(obj, 1) != 0;
    }

    private int next(T obj, int bits) {
        long prevValue, nextValue;
        do {
            prevValue = get(obj);
            nextValue = (prevValue * multiplier + addend) & mask;
        } while (!compareAndSet(obj, prevValue, nextValue));

        return (int) (nextValue >>> (48 - bits));
    }

    // ---- ATOMIC LONG ----

    private boolean compareAndSet(T obj, long expect, long update) {
        if (obj == null || obj.getClass() != targetClass) {
            checkInstanceClass(obj);
        }
        return unsafe.compareAndSwapLong(obj, offset, expect, update);
    }

    private long get(T obj) {
        if (obj == null || obj.getClass() != targetClass) {
            checkInstanceClass(obj);
        }
        return unsafe.getLongVolatile(obj, offset);
    }

    private void set(T obj, long value) {
        if (obj == null || obj.getClass() != targetClass) {
            checkInstanceClass(obj);
        }
        unsafe.putLongVolatile(obj, offset, value);
    }

    private void checkInstanceClass(T obj) {
        if (!targetClass.isInstance(obj)) {
            String msg = String.format("object %s is not instance of %s", String.valueOf(obj), targetClass.getName());
            throw new ClassCastException(msg);
        }
    }
}
