package ru.yandex.solomon.util.concurrent;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Autoreset event
 *
 * @url https://msdn.microsoft.com/en-us/library/windows/desktop/ms682655%28v=vs.85%29.aspx
 * @author Stepan Koltsov
 */
public class AutoresetEvent {

    private final Lock lock = new ReentrantLock();
    private final Condition condition = lock.newCondition();


    private boolean signalled = false;

    public void signal() {
        lock.lock();
        try {
            signalled = true;
            condition.signal();
        } finally {
            lock.unlock();
        }
    }

    public void await() {
        lock.lock();
        try {
            while (!signalled) {
                try {
                    condition.await();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
            signalled = false;
        } finally {
            lock.unlock();
        }
    }

    public boolean tryAwait() {
        return awaitMillis(0);
    }

    public boolean awaitMillis(long millis) {
        return await(millis, TimeUnit.MILLISECONDS);
    }

    public boolean await(long time, TimeUnit unit) {
        lock.lock();
        try {
            if (signalled) {
                signalled = false;
                return true;
            }
            while (!signalled) {
                try {
                    if (!condition.await(time, unit)) {
                        return false;
                    }
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
            signalled = false;
            return true;
        } finally {
            lock.unlock();
        }
    }

    public boolean awaitUntil(long instant) {
        long now = System.currentTimeMillis();
        if (now >= instant) {
            return false;
        }
        return awaitMillis(instant - now);
    }
}
