package ru.yandex.solomon.util.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.util.io.IoBiConsumer;
import ru.yandex.solomon.util.io.IoFunction;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;
import static java.nio.file.StandardOpenOption.WRITE;
import static java.util.stream.Collectors.toList;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class SimpleFileStorage implements FileStorage {

    private static final Logger logger = LoggerFactory.getLogger(SimpleFileStorage.class);

    private final Path storage;

    public SimpleFileStorage(Path storage) {
        createDirIfNecessary(storage);
        this.storage = storage;
    }

    @Override
    public <V> void save(String fileName, V value, IoFunction<V, String> serialize) throws Exception {
        writeToFile(
            fileName,
            value,
            (tmp, v) -> Files.writeString(tmp, serialize.apply(v), CREATE, WRITE, TRUNCATE_EXISTING));
    }

    @Override
    @Nullable
    public <V> V load(String fileName, IoFunction<String, V> deserialize) throws Exception {
        return readFromFile(
            fileName,
            target -> deserialize.apply(Files.readString(target)));
    }

    @Override
    public <V> void saveValues(
        String fileName,
        Collection<V> values,
        IoFunction<V, String> serializeOne) throws Exception
    {
        writeToFile(
            fileName,
            values,
            (tmp, vs) -> {
                try (var writer = Files.newBufferedWriter(tmp, CREATE, WRITE, TRUNCATE_EXISTING)) {
                    for (var v : vs) {
                        writer.write(serializeOne.apply(v));
                        writer.newLine();
                    }
                }
            });
    }

    @Override
    @Nullable
    public <V> List<V> loadValues(String fileName, IoFunction<String, V> deserializeOne) throws Exception {
        return readFromFile(
            fileName,
            target -> {
                try (var lines = Files.lines(target)) {
                    return lines
                        .map(deserializeOne::applyUnchecked)
                        .filter(Objects::nonNull)
                        .collect(toList());
                }
            });
    }

    private <T> void writeToFile(String fileName, T payload, IoBiConsumer<Path, T> writer) throws Exception {
        if (!Files.isWritable(storage)) {
            return;
        }

        var target = storage.resolve(fileName);
        var tmp = storage.resolve("tmp." + fileName);

        writer.accept(tmp, payload);
        Files.move(tmp, target, REPLACE_EXISTING);
    }

    @Nullable
    private  <T> T readFromFile(String fileName, IoFunction<Path, T> reader) throws Exception {
        var target = storage.resolve(fileName);
        if (!Files.isReadable(target)) {
            return null;
        }

        return reader.apply(target);
    }

    private static void createDirIfNecessary(Path storage) {
        try {
            if (Files.exists(storage)) {
                return;
            }

            Files.createDirectory(storage);
        } catch (IOException e) {
            logger.error("Unable create file storage dir: " + storage, e);
        }
    }
}
