package ru.yandex.solomon.util.future;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class RetryCompletableFuture {
    private static final ScheduledExecutorService scheduler = new ScheduledThreadPoolExecutor(1);
    private RetryCompletableFuture() {
    }

    public static <T> CompletableFuture<T> runWithRetries(Supplier<CompletableFuture<T>> taskProducer) {
        return runWithRetries(taskProducer, RetryConfig.DEFAULT);
    }

    public static <T> CompletableFuture<T> runWithRetries(
        Supplier<CompletableFuture<T>> taskProducer,
        RetryConfig config)
    {
        return new Task<>(taskProducer, config).run();
    }

    private static class Task<T> extends CompletableFuture<T> {
        private final Supplier<CompletableFuture<T>> supplier;
        private final RetryConfig config;
        private int attempt;
        private long startNanos;
        private volatile ScheduledFuture<?> scheduled;

        public Task(Supplier<CompletableFuture<T>> supplier, RetryConfig config) {
            this.supplier = supplier;
            this.config = config;
        }

        public CompletableFuture<T> run() {
            call();
            return this;
        }

        private void call() {
            try {
                if (super.isDone()) {
                    return;
                }

                startNanos = System.nanoTime();
                supplier.get().whenComplete(this::whenComplete);
            } catch (Throwable e) {
                completeExceptionally(e);
            }
        }

        private void whenComplete(T result, @Nullable Throwable throwable) {
            try {
                if (throwable == null) {
                    complete(result);
                    return;
                }

                if (!config.getExceptionFilter().test(throwable)) {
                    completeExceptionally(throwable);
                    return;
                }

                if (++attempt >= config.getNumRetries()) {
                    completeExceptionally(throwable);
                    return;
                }

                long timeSpentMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                config.getRetryStats().addFailure(timeSpentMillis, throwable);
                scheduled = schedule(backoffTimeMillis());
            } catch (Throwable e) {
                completeExceptionally(e);
            }
        }

        private ScheduledFuture<?> schedule(long delayMillis) {
            return scheduler.schedule(this::call, delayMillis, TimeUnit.MILLISECONDS);
        }

        private long backoffTimeMillis() {
            int slots = 1 << attempt;
            long delayMillis = Math.min(config.getRetryDelayMillis() * slots, config.getMaxRetryDelayMillis());
            return randomizeDelay(delayMillis);
        }

        private long randomizeDelay(long delayMillis) {
            if (delayMillis == 0) {
                return 0;
            }

            var half = Math.max(delayMillis / 2, 1);
            return half + ThreadLocalRandom.current().nextLong(half);
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            boolean result = super.cancel(mayInterruptIfRunning);
            var copy = scheduled;
            if (copy != null) {
                copy.cancel(mayInterruptIfRunning);
            }
            return result;
        }
    }
}
