package ru.yandex.solomon.util.future;

import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class RetryConfig implements Cloneable {
    public static final RetryConfig DEFAULT = new RetryConfig(
        3,
        RetryStats.NOP,
        x -> true,
        0,
            0
    );

    private int numRetries;
    private RetryStats retryStats;
    private Predicate<Throwable> exceptionFilter;
    private long retryDelayMillis;
    private long maxRetryDelayMillis;

    public RetryStats getRetryStats() {
        return retryStats;
    }

    public int getNumRetries() {
        return numRetries;
    }

    public Predicate<Throwable> getExceptionFilter() {
        return exceptionFilter;
    }

    public long getRetryDelayMillis() {
        return retryDelayMillis;
    }

    public long getMaxRetryDelayMillis() {
        return maxRetryDelayMillis;
    }

    private RetryConfig(int numRetries, RetryStats retryStats, Predicate<Throwable> exceptionFilter, long retryDelayMillis, long maxRetryDelayMillis) {
        this.numRetries = numRetries;
        this.retryStats = retryStats;
        this.exceptionFilter = exceptionFilter;
        this.retryDelayMillis = retryDelayMillis;
        this.maxRetryDelayMillis = maxRetryDelayMillis;
    }

    /**
     * Builds a clone of current config with custom retry limit
     * @param numRetries how many tries we should retry, if failures occur (default: 3)
     */
    public RetryConfig withNumRetries(int numRetries) {
        RetryConfig result = safeClone();
        result.numRetries = numRetries;
        return result;
    }

    /**
     * Builds a clone of current config with custom retry stats collector
     * @param stats custom retry stats collector (default: {@link RetryStats#NOP} - nop implementation)
     */
    public RetryConfig withStats(RetryStats stats) {
        RetryConfig result = safeClone();
        result.retryStats = stats;
        return result;
    }

    /**
     * Builds a clone of current config with custom exception filter (which defines which execution exceptions lead to new retries - and which prevent engine from new retries)
     * Clarification: if this predicate returns true on an exception, a new try will be attempted. Otherwise retries will be terminated.
     * @param exceptionFilter custom exception filter (default: all exceptions lead to further retries)
     */
    public RetryConfig withExceptionFilter(Predicate<Throwable> exceptionFilter) {
        RetryConfig result = safeClone();
        result.exceptionFilter = exceptionFilter;
        return result;
    }

    /**
     * Builds a clone of current config with custom retry delay
     * @param retryDelayMillis how long should we wait until nex retry (default: no pause at all)
     */
    public RetryConfig withDelay(long retryDelayMillis) {
        RetryConfig result = safeClone();
        result.retryDelayMillis = retryDelayMillis;
        result.maxRetryDelayMillis = retryDelayMillis;
        return result;
    }

    public RetryConfig withMaxDelay(long maxRetryDelayMillis) {
        RetryConfig result = safeClone();
        result.maxRetryDelayMillis = maxRetryDelayMillis;
        return result;
    }

    private RetryConfig safeClone() {
        try {
            return (RetryConfig) clone();
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }
}
