package ru.yandex.solomon.util.future;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class RetryUtils {

    private static class InvocationHandlerWithRetries<T> implements InvocationHandler {
        private final RetryConfig config;
        private final T target;

        private InvocationHandlerWithRetries(T target, RetryConfig config) {
            this.target = target;
            this.config = config;
        }

        @SuppressWarnings("unchecked")
        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if (method.getReturnType() != CompletableFuture.class) {
                return method.invoke(target, args);
            }
            return RetryCompletableFuture.runWithRetries(
                    () -> {
                        try {
                            return (CompletableFuture) method.invoke(target, args);
                        } catch (IllegalAccessException | InvocationTargetException e) {
                            throw new RuntimeException(e);
                        }
                    },
                    config
            );
        }
    }

    /**
     * This method accepts an interface and an object, implementing this interface.
     * As a result, this method generates an object, implementing given interface. All its methods calls are proxied to
     * given object, except for the methods returning CompletableFutures.
     * Such methods are wrapped using RetryCompletableFuture.
     * @param target object to wrap
     * @param iface interface we should generate
     * @param config Config for the retry engine
     * @return resulting proxy
     */
    public static <T> T wrapWithRetries(
            T target,
            Class<T> iface,
            RetryConfig config)
    {
        InvocationHandlerWithRetries<T> handler = new InvocationHandlerWithRetries<>(
                target,
                config
        );
        return (T) Proxy.newProxyInstance(
                iface.getClassLoader(),
                new Class[]{iface},
                handler);
    }

}
