package ru.yandex.solomon.util.host;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class DefaultDnsResolver implements DnsResolver {

    private static final Logger logger = LoggerFactory.getLogger(DefaultDnsResolver.class);

    @Override
    public CompletableFuture<Inet6Address> resolve(String fqdn) {
        return CompletableFuture.completedFuture(resolveSync(fqdn));
    }

    private Inet6Address resolveSync(String fqdn) {
        try {
            return Arrays.stream(InetAddress.getAllByName(fqdn))
                    .filter(addr -> addr instanceof Inet6Address)
                    .map(addr -> (Inet6Address) addr)
                    .findFirst()
                    .orElseGet(() -> {
                        logger.error("Failed to find ipv6 address for " + fqdn);
                        return null;
                    });
        } catch (UnknownHostException e) {
            logger.error("Host is unknown", e);
            return null;
        }
    }
}
