package ru.yandex.solomon.util.http;

import java.net.InetSocketAddress;
import java.net.URI;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import org.springframework.http.server.reactive.ServerHttpRequest;


/**
 * @author Sergey Polovko
 */
public final class HttpUtils {
    private HttpUtils() {}

    @Nullable
    public static String cookieValue(ServerHttpRequest request, String name) {
        String cookieHeader = request.getHeaders().getFirst("Cookie");
        if (cookieHeader == null) {
            return null;
        }
        return cookieValue(cookieHeader, name);
    }

    @Nullable
    public static String cookieValue(String cookieHeader, String name) {
        Set<Cookie> cookies = ServerCookieDecoder.LAX.decode(cookieHeader);
        for (Cookie cookie : cookies) {
            if (name.equals(cookie.name())) {
                return cookie.value();
            }
        }
        return null;
    }

    public static String realOrRemoteIp(ServerHttpRequest request) {
        {
            // X-Forwarded-For may contain comma-separated list of addresses
            // if there's more than one proxy involved, e. g.
            // x-forwarded-for: 2a02:6b8:0:40c:2d1e:e4e5:1335:7b59, 87.250.246.211

            String value = request.getHeaders().getFirst("X-Forwarded-For");
            if (value != null && !value.isEmpty()) {
                int commaIdx = value.indexOf(',');
                if (commaIdx > 0) {
                    return value.substring(0, commaIdx);
                }
                return value;
            }
        }
        {
            String value = request.getHeaders().getFirst("X-Real-IP");
            if (value != null && !value.isEmpty()) {
                return value;
            }
        }

        InetSocketAddress remoteAddress = request.getRemoteAddress();
        return remoteAddress != null ? remoteAddress.getAddress().toString() : "unknown";
    }

    public static String requestId(ServerHttpRequest request) {
        var headers = request.getHeaders();
        var result = headers.getFirst("X-Request-Id");
        if (Strings.isNullOrEmpty(result)) {
            result = headers.getFirst("x-request-id");
        }
        return Strings.nullToEmpty(result);
    }

    public static String requestUri(ServerHttpRequest req) {
        URI uri = req.getURI();
        String rawQuery = uri.getRawQuery();
        return (rawQuery == null) ? uri.getPath() : uri.getPath() + '?' + rawQuery;
    }
}
