package ru.yandex.solomon.util.jackson;

import java.io.IOException;
import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

/**
 * Deserializer for Instant fields.
 * Supports ISO-8601 format and UNIX-epoch milliseconds ("2018-07-11T11:28:52.965" or 1531308532965)
 *
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class InstantDeserializer extends StdDeserializer<Instant> {

    public InstantDeserializer() {
        super(Instant.class);
    }

    @Override
    public Instant deserialize(JsonParser parser, DeserializationContext context) throws IOException {
        JsonToken token = parser.getCurrentToken();

        if (token == JsonToken.VALUE_NUMBER_INT) {
            long number = parser.getLongValue();
            return Instant.ofEpochMilli(number);
        } else if (token == JsonToken.VALUE_STRING) {
            String text = parser.getValueAsString();
            try {
                return Instant.parse(text);
            } catch (Exception ignored) {
                throw context.mappingException("failed to parse time in ISO-8601 format");
            }
        }

        throw context.wrongTokenException(parser, token, "failed to read time: expected UNIX-epoch milliseconds or ISO-8601-formatted text");
    }
}
