package ru.yandex.solomon.util.jackson;

import java.io.FileDescriptor;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.Map;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.www.ser.WwwDurationSerializer;


/**
 * @author Stepan Koltsov
 */
public class SalmonJackson {


    private static Module timeModule() {
        SimpleModule module = new SimpleModule();
        module.addSerializer(Instant.class, new JsonSerializer<Instant>() {
            @Override
            public void serialize(Instant value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
                jgen.writeString(value.toString());
            }
        });
        module.addDeserializer(Instant.class, new JsonDeserializer<Instant>() {
            @Override
            public Instant deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
                String instantSerialized = p.readValueAs(String.class);
                return Instant.parse(instantSerialized);
            }
        });
        module.addDeserializer(Instant.class, new InstantDeserializer());
        module.addSerializer(Duration.class, new JsonSerializer<Duration>() {
            @Override
            public void serialize(Duration value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
                jgen.writeString(WwwDurationSerializer.formatDuration(value));
            }
        });
        module.addSerializer(LocalDate.class, new JsonSerializer<LocalDate>() {
            @Override
            public void serialize(LocalDate value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
                jgen.writeString(value.toString());
            }
        });
        module.addSerializer(Labels.class, new JsonSerializer<Labels>() {
            @Override
            public void serialize(Labels labels, JsonGenerator jgen, SerializerProvider provider) throws IOException {
                jgen.writeStartObject();
                labels.forEach(l -> {
                    try {
                        jgen.writeStringField(l.getKey(), l.getValue());
                    } catch (IOException e) {
                        throw ExceptionUtils.throwException(e);
                    }
                });
                jgen.writeEndObject();
            }
        });

        // protobuf
        module.addSerializer(Descriptors.FileDescriptor.class, new JsonSerializer<Descriptors.FileDescriptor>() {
            @Override
            public void serialize(Descriptors.FileDescriptor value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
                throw new IllegalStateException(
                        FileDescriptor.class + " is not supposed to be serialized");
            }
        });
        module.addSerializer(Message.class, new JsonSerializer<Message>() {
            @Override
            public void serialize(Message value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
                Map<String, Object> pojo = WwwProtobuf.toJson(value);
                jgen.writeObject(pojo);
            }
        });

        return module;
    }

    private static final ObjectMapper objectMapper;

    static {
        objectMapper = new ObjectMapper();
        objectMapper.registerModule(timeModule());
    }

    public static ObjectMapper getObjectMapper() {
        return objectMapper;
    }
}
